/*
 * Decompiled with CFR 0.152.
 */
package org.apache.commons.compress.archivers.tar;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.Calendar;
import java.util.Date;
import java.util.Map;
import java.util.TimeZone;
import java.util.zip.GZIPInputStream;
import org.apache.commons.compress.AbstractTestCase;
import org.apache.commons.compress.archivers.ArchiveEntry;
import org.apache.commons.compress.archivers.tar.TarArchiveEntry;
import org.apache.commons.compress.archivers.tar.TarArchiveInputStream;
import org.apache.commons.compress.archivers.tar.TarArchiveOutputStream;
import org.apache.commons.compress.utils.IOUtils;
import org.junit.Assert;
import org.junit.Test;

public class TarArchiveInputStreamTest {
    @Test
    public void readSimplePaxHeader() throws Exception {
        ByteArrayInputStream is = new ByteArrayInputStream(new byte[1]);
        TarArchiveInputStream tais = new TarArchiveInputStream((InputStream)is);
        Map headers = tais.parsePaxHeaders((InputStream)new ByteArrayInputStream("30 atime=1321711775.972059463\n".getBytes("UTF-8")));
        Assert.assertEquals((long)1L, (long)headers.size());
        Assert.assertEquals((Object)"1321711775.972059463", headers.get("atime"));
        tais.close();
    }

    @Test
    public void secondEntryWinsWhenPaxHeaderContainsDuplicateKey() throws Exception {
        ByteArrayInputStream is = new ByteArrayInputStream(new byte[1]);
        TarArchiveInputStream tais = new TarArchiveInputStream((InputStream)is);
        Map headers = tais.parsePaxHeaders((InputStream)new ByteArrayInputStream("11 foo=bar\n11 foo=baz\n".getBytes("UTF-8")));
        Assert.assertEquals((long)1L, (long)headers.size());
        Assert.assertEquals((Object)"baz", headers.get("foo"));
        tais.close();
    }

    @Test
    public void paxHeaderEntryWithEmptyValueRemovesKey() throws Exception {
        ByteArrayInputStream is = new ByteArrayInputStream(new byte[1]);
        TarArchiveInputStream tais = new TarArchiveInputStream((InputStream)is);
        Map headers = tais.parsePaxHeaders((InputStream)new ByteArrayInputStream("11 foo=bar\n7 foo=\n".getBytes("UTF-8")));
        Assert.assertEquals((long)0L, (long)headers.size());
        tais.close();
    }

    @Test
    public void readPaxHeaderWithEmbeddedNewline() throws Exception {
        ByteArrayInputStream is = new ByteArrayInputStream(new byte[1]);
        TarArchiveInputStream tais = new TarArchiveInputStream((InputStream)is);
        Map headers = tais.parsePaxHeaders((InputStream)new ByteArrayInputStream("28 comment=line1\nline2\nand3\n".getBytes("UTF-8")));
        Assert.assertEquals((long)1L, (long)headers.size());
        Assert.assertEquals((Object)"line1\nline2\nand3", headers.get("comment"));
        tais.close();
    }

    @Test
    public void readNonAsciiPaxHeader() throws Exception {
        String ae = "\u00e4";
        String line = "11 path=\u00e4\n";
        Assert.assertEquals((long)11L, (long)"11 path=\u00e4\n".getBytes("UTF-8").length);
        ByteArrayInputStream is = new ByteArrayInputStream(new byte[1]);
        TarArchiveInputStream tais = new TarArchiveInputStream((InputStream)is);
        Map headers = tais.parsePaxHeaders((InputStream)new ByteArrayInputStream("11 path=\u00e4\n".getBytes("UTF-8")));
        Assert.assertEquals((long)1L, (long)headers.size());
        Assert.assertEquals((Object)"\u00e4", headers.get("path"));
        tais.close();
    }

    @Test
    public void workaroundForBrokenTimeHeader() throws Exception {
        try (TarArchiveInputStream in = null;){
            in = new TarArchiveInputStream((InputStream)new FileInputStream(AbstractTestCase.getFile("simple-aix-native-tar.tar")));
            TarArchiveEntry tae = in.getNextTarEntry();
            tae = in.getNextTarEntry();
            Assert.assertEquals((Object)"sample/link-to-txt-file.lnk", (Object)tae.getName());
            Assert.assertEquals((Object)new Date(0L), (Object)tae.getLastModifiedDate());
            Assert.assertTrue((boolean)tae.isSymbolicLink());
            Assert.assertTrue((boolean)tae.isCheckSumOK());
        }
    }

    @Test
    public void datePriorToEpochInGNUFormat() throws Exception {
        this.datePriorToEpoch("preepoch-star.tar");
    }

    @Test
    public void datePriorToEpochInPAXFormat() throws Exception {
        this.datePriorToEpoch("preepoch-posix.tar");
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void datePriorToEpoch(String archive) throws Exception {
        try (TarArchiveInputStream in = null;){
            in = new TarArchiveInputStream((InputStream)new FileInputStream(AbstractTestCase.getFile(archive)));
            TarArchiveEntry tae = in.getNextTarEntry();
            Assert.assertEquals((Object)"foo", (Object)tae.getName());
            Calendar cal = Calendar.getInstance(TimeZone.getTimeZone("GMT"));
            cal.set(1969, 11, 31, 23, 59, 59);
            cal.set(14, 0);
            Assert.assertEquals((Object)cal.getTime(), (Object)tae.getLastModifiedDate());
            Assert.assertTrue((boolean)tae.isCheckSumOK());
        }
    }

    @Test
    public void testCompress197() throws Exception {
        try (TarArchiveInputStream tar = this.getTestStream("/COMPRESS-197.tar");){
            TarArchiveEntry entry = tar.getNextTarEntry();
            while (entry != null) {
                entry = tar.getNextTarEntry();
            }
        }
        catch (IOException e) {
            Assert.fail((String)("COMPRESS-197: " + e.getMessage()));
        }
    }

    @Test
    public void shouldUseSpecifiedEncodingWhenReadingGNULongNames() throws Exception {
        ByteArrayOutputStream bos = new ByteArrayOutputStream();
        String encoding = "UTF-16";
        String name = "12345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890\u00e4";
        TarArchiveOutputStream tos = new TarArchiveOutputStream((OutputStream)bos, "UTF-16");
        tos.setLongFileMode(2);
        TarArchiveEntry t = new TarArchiveEntry("12345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890\u00e4");
        t.setSize(1L);
        tos.putArchiveEntry((ArchiveEntry)t);
        tos.write(30);
        tos.closeArchiveEntry();
        tos.close();
        byte[] data = bos.toByteArray();
        ByteArrayInputStream bis = new ByteArrayInputStream(data);
        TarArchiveInputStream tis = new TarArchiveInputStream((InputStream)bis, "UTF-16");
        t = tis.getNextTarEntry();
        Assert.assertEquals((Object)"12345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890\u00e4", (Object)t.getName());
        tis.close();
    }

    @Test
    public void shouldConsumeArchiveCompletely() throws Exception {
        InputStream is = TarArchiveInputStreamTest.class.getResourceAsStream("/archive_with_trailer.tar");
        TarArchiveInputStream tar = new TarArchiveInputStream(is);
        while (tar.getNextTarEntry() != null) {
        }
        byte[] expected = new byte[]{72, 101, 108, 108, 111, 44, 32, 119, 111, 114, 108, 100, 33, 10};
        byte[] actual = new byte[expected.length];
        is.read(actual);
        Assert.assertArrayEquals((byte[])expected, (byte[])actual);
        tar.close();
    }

    @Test
    public void readsArchiveCompletely_COMPRESS245() throws Exception {
        try (InputStream is = TarArchiveInputStreamTest.class.getResourceAsStream("/COMPRESS-245.tar.gz");){
            GZIPInputStream gin = new GZIPInputStream(is);
            TarArchiveInputStream tar = new TarArchiveInputStream((InputStream)gin);
            int count = 0;
            TarArchiveEntry entry = tar.getNextTarEntry();
            while (entry != null) {
                ++count;
                entry = tar.getNextTarEntry();
            }
            Assert.assertEquals((long)31L, (long)count);
            tar.close();
        }
        catch (IOException e) {
            Assert.fail((String)("COMPRESS-245: " + e.getMessage()));
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test(expected=IOException.class)
    public void shouldThrowAnExceptionOnTruncatedEntries() throws Exception {
        File dir = AbstractTestCase.mkdir("COMPRESS-279");
        TarArchiveInputStream is = this.getTestStream("/COMPRESS-279.tar");
        FileOutputStream out = null;
        try {
            TarArchiveEntry entry = is.getNextTarEntry();
            int count = 0;
            while (entry != null) {
                out = new FileOutputStream(new File(dir, String.valueOf(count)));
                IOUtils.copy((InputStream)is, (OutputStream)out);
                out.close();
                out = null;
                ++count;
                entry = is.getNextTarEntry();
            }
        }
        finally {
            is.close();
            if (out != null) {
                out.close();
            }
            AbstractTestCase.rmdir(dir);
        }
    }

    @Test
    public void shouldReadBigGid() throws Exception {
        ByteArrayOutputStream bos = new ByteArrayOutputStream();
        TarArchiveOutputStream tos = new TarArchiveOutputStream((OutputStream)bos);
        tos.setBigNumberMode(2);
        TarArchiveEntry t = new TarArchiveEntry("name");
        t.setGroupId(0xFFFFFFFEL);
        t.setSize(1L);
        tos.putArchiveEntry((ArchiveEntry)t);
        tos.write(30);
        tos.closeArchiveEntry();
        tos.close();
        byte[] data = bos.toByteArray();
        ByteArrayInputStream bis = new ByteArrayInputStream(data);
        TarArchiveInputStream tis = new TarArchiveInputStream((InputStream)bis);
        t = tis.getNextTarEntry();
        Assert.assertEquals((long)0xFFFFFFFEL, (long)t.getLongGroupId());
        tis.close();
    }

    @Test
    public void shouldReadGNULongNameEntryWithWrongName() throws Exception {
        try (TarArchiveInputStream is = this.getTestStream("/COMPRESS-324.tar");){
            TarArchiveEntry entry = is.getNextTarEntry();
            Assert.assertEquals((Object)"1234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890.txt", (Object)entry.getName());
        }
    }

    @Test
    public void survivesBlankLinesInPaxHeader() throws Exception {
        try (TarArchiveInputStream is = this.getTestStream("/COMPRESS-355.tar");){
            TarArchiveEntry entry = is.getNextTarEntry();
            Assert.assertEquals((Object)"package/package.json", (Object)entry.getName());
            Assert.assertNull((Object)is.getNextTarEntry());
        }
    }

    @Test
    public void survivesPaxHeaderWithNameEndingInSlash() throws Exception {
        try (TarArchiveInputStream is = this.getTestStream("/COMPRESS-356.tar");){
            TarArchiveEntry entry = is.getNextTarEntry();
            Assert.assertEquals((Object)"package/package.json", (Object)entry.getName());
            Assert.assertNull((Object)is.getNextTarEntry());
        }
    }

    @Test
    public void testGetAndSetOfPaxEntry() throws Exception {
        try (TarArchiveInputStream is = this.getTestStream("/COMPRESS-356.tar");){
            TarArchiveEntry entry = is.getNextTarEntry();
            Assert.assertEquals((Object)"package/package.json", (Object)entry.getName());
            Assert.assertEquals((Object)is.getCurrentEntry(), (Object)entry);
            TarArchiveEntry weaselEntry = new TarArchiveEntry(entry.getName());
            weaselEntry.setSize(entry.getSize());
            is.setCurrentEntry(weaselEntry);
            Assert.assertEquals((Object)entry, (Object)is.getCurrentEntry());
            Assert.assertFalse((entry == is.getCurrentEntry() ? 1 : 0) != 0);
            Assert.assertTrue((weaselEntry == is.getCurrentEntry() ? 1 : 0) != 0);
            try {
                is.setCurrentEntry(null);
                is.read();
                Assert.fail((String)"should abort because current entry is nulled");
            }
            catch (IllegalStateException illegalStateException) {
                // empty catch block
            }
            is.setCurrentEntry(entry);
            is.read();
        }
    }

    @Test
    public void skipsDevNumbersWhenEntryIsNoDevice() throws Exception {
        try (TarArchiveInputStream is = this.getTestStream("/COMPRESS-417.tar");){
            Assert.assertEquals((Object)"test1.xml", (Object)is.getNextTarEntry().getName());
            Assert.assertEquals((Object)"test2.xml", (Object)is.getNextTarEntry().getName());
            Assert.assertNull((Object)is.getNextTarEntry());
        }
    }

    @Test
    public void singleByteReadConsistentlyReturnsMinusOneAtEof() throws Exception {
        try (FileInputStream in = new FileInputStream(AbstractTestCase.getFile("bla.tar"));
             TarArchiveInputStream archive = new TarArchiveInputStream((InputStream)in);){
            ArchiveEntry e = archive.getNextEntry();
            IOUtils.toByteArray((InputStream)archive);
            Assert.assertEquals((long)-1L, (long)archive.read());
            Assert.assertEquals((long)-1L, (long)archive.read());
        }
    }

    @Test
    public void multiByteReadConsistentlyReturnsMinusOneAtEof() throws Exception {
        byte[] buf = new byte[2];
        try (FileInputStream in = new FileInputStream(AbstractTestCase.getFile("bla.tar"));
             TarArchiveInputStream archive = new TarArchiveInputStream((InputStream)in);){
            ArchiveEntry e = archive.getNextEntry();
            IOUtils.toByteArray((InputStream)archive);
            Assert.assertEquals((long)-1L, (long)archive.read(buf));
            Assert.assertEquals((long)-1L, (long)archive.read(buf));
        }
    }

    private TarArchiveInputStream getTestStream(String name) {
        return new TarArchiveInputStream(TarArchiveInputStreamTest.class.getResourceAsStream(name));
    }
}

