/*
 * Decompiled with CFR 0.152.
 */
package org.apache.commons.io;

import java.io.BufferedOutputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileFilter;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.math.BigInteger;
import java.net.URL;
import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.GregorianCalendar;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.zip.CRC32;
import java.util.zip.Checksum;
import org.apache.commons.io.Charsets;
import org.apache.commons.io.DirectoryWalker;
import org.apache.commons.io.FileExistsException;
import org.apache.commons.io.FileUtils;
import org.apache.commons.io.FilenameUtils;
import org.apache.commons.io.IOUtils;
import org.apache.commons.io.filefilter.IOFileFilter;
import org.apache.commons.io.filefilter.NameFileFilter;
import org.apache.commons.io.filefilter.WildcardFileFilter;
import org.apache.commons.io.testtools.TestUtils;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Ignore;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.TemporaryFolder;

public class FileUtilsTestCase {
    @Rule
    public TemporaryFolder temporaryFolder = new TemporaryFolder();
    private static final int TEST_DIRECTORY_SIZE = 0;
    private static final BigInteger TEST_DIRECTORY_SIZE_BI = BigInteger.ZERO;
    private static final BigInteger TEST_DIRECTORY_SIZE_GT_ZERO_BI = BigInteger.valueOf(100L);
    private static final ListDirectoryWalker LIST_WALKER = new ListDirectoryWalker();
    private File testFile1;
    private File testFile2;
    private int testFile1Size;
    private int testFile2Size;

    private File getTestDirectory() {
        return this.temporaryFolder.getRoot();
    }

    @Before
    public void setUp() throws Exception {
        this.testFile1 = new File(this.getTestDirectory(), "file1-test.txt");
        this.testFile2 = new File(this.getTestDirectory(), "file1a-test.txt");
        this.testFile1Size = (int)this.testFile1.length();
        this.testFile2Size = (int)this.testFile2.length();
        if (!this.testFile1.getParentFile().exists()) {
            throw new IOException("Cannot create file " + this.testFile1 + " as the parent directory does not exist");
        }
        try (BufferedOutputStream output3 = new BufferedOutputStream(new FileOutputStream(this.testFile1));){
            TestUtils.generateTestData(output3, this.testFile1Size);
        }
        if (!this.testFile2.getParentFile().exists()) {
            throw new IOException("Cannot create file " + this.testFile2 + " as the parent directory does not exist");
        }
        var2_2 = null;
        try (BufferedOutputStream output2 = new BufferedOutputStream(new FileOutputStream(this.testFile2));){
            TestUtils.generateTestData(output2, this.testFile2Size);
        }
        catch (Throwable throwable) {
            var2_2 = throwable;
            throw throwable;
        }
        FileUtils.deleteDirectory((File)this.getTestDirectory());
        this.getTestDirectory().mkdirs();
        if (!this.testFile1.getParentFile().exists()) {
            throw new IOException("Cannot create file " + this.testFile1 + " as the parent directory does not exist");
        }
        var2_2 = null;
        try (BufferedOutputStream output1 = new BufferedOutputStream(new FileOutputStream(this.testFile1));){
            TestUtils.generateTestData(output1, this.testFile1Size);
        }
        catch (Throwable throwable) {
            var2_2 = throwable;
            throw throwable;
        }
        if (!this.testFile2.getParentFile().exists()) {
            throw new IOException("Cannot create file " + this.testFile2 + " as the parent directory does not exist");
        }
        var2_2 = null;
        try (BufferedOutputStream output = new BufferedOutputStream(new FileOutputStream(this.testFile2));){
            TestUtils.generateTestData(output, this.testFile2Size);
        }
        catch (Throwable throwable) {
            var2_2 = throwable;
            throw throwable;
        }
    }

    private String getName() {
        return this.getClass().getSimpleName();
    }

    @Test
    public void testGetFile() {
        File expected_A = new File("src");
        File expected_B = new File(expected_A, "main");
        File expected_C = new File(expected_B, "java");
        Assert.assertEquals((String)"A", (Object)expected_A, (Object)FileUtils.getFile((String[])new String[]{"src"}));
        Assert.assertEquals((String)"B", (Object)expected_B, (Object)FileUtils.getFile((String[])new String[]{"src", "main"}));
        Assert.assertEquals((String)"C", (Object)expected_C, (Object)FileUtils.getFile((String[])new String[]{"src", "main", "java"}));
        try {
            FileUtils.getFile((String[])null);
            Assert.fail((String)"Expected NullPointerException");
        }
        catch (NullPointerException nullPointerException) {
            // empty catch block
        }
    }

    @Test
    public void testGetFile_Parent() {
        File parent = new File("parent");
        File expected_A = new File(parent, "src");
        File expected_B = new File(expected_A, "main");
        File expected_C = new File(expected_B, "java");
        Assert.assertEquals((String)"A", (Object)expected_A, (Object)FileUtils.getFile((File)parent, (String[])new String[]{"src"}));
        Assert.assertEquals((String)"B", (Object)expected_B, (Object)FileUtils.getFile((File)parent, (String[])new String[]{"src", "main"}));
        Assert.assertEquals((String)"C", (Object)expected_C, (Object)FileUtils.getFile((File)parent, (String[])new String[]{"src", "main", "java"}));
        try {
            FileUtils.getFile((File)parent, (String[])null);
            Assert.fail((String)"Expected NullPointerException");
        }
        catch (NullPointerException nullPointerException) {
            // empty catch block
        }
        try {
            FileUtils.getFile((File)null, (String[])new String[]{"src"});
            Assert.fail((String)"Expected NullPointerException");
        }
        catch (NullPointerException nullPointerException) {
            // empty catch block
        }
    }

    @Test
    public void testGetTempDirectoryPath() {
        Assert.assertEquals((Object)System.getProperty("java.io.tmpdir"), (Object)FileUtils.getTempDirectoryPath());
    }

    @Test
    public void testGetTempDirectory() {
        File tempDirectory = new File(System.getProperty("java.io.tmpdir"));
        Assert.assertEquals((Object)tempDirectory, (Object)FileUtils.getTempDirectory());
    }

    @Test
    public void testGetUserDirectoryPath() {
        Assert.assertEquals((Object)System.getProperty("user.home"), (Object)FileUtils.getUserDirectoryPath());
    }

    @Test
    public void testGetUserDirectory() {
        File userDirectory = new File(System.getProperty("user.home"));
        Assert.assertEquals((Object)userDirectory, (Object)FileUtils.getUserDirectory());
    }

    @Test
    public void test_openInputStream_exists() throws Exception {
        File file = new File(this.getTestDirectory(), "test.txt");
        TestUtils.createLineBasedFile(file, new String[]{"Hello"});
        try (FileInputStream in = FileUtils.openInputStream((File)file);){
            Assert.assertEquals((long)72L, (long)in.read());
        }
    }

    @Test
    public void test_openInputStream_existsButIsDirectory() throws Exception {
        File directory = new File(this.getTestDirectory(), "subdir");
        directory.mkdirs();
        try (FileInputStream in = FileUtils.openInputStream((File)directory);){
            Assert.fail();
        }
        catch (IOException iOException) {
            // empty catch block
        }
    }

    @Test
    public void test_openInputStream_notExists() throws Exception {
        File directory = new File(this.getTestDirectory(), "test.txt");
        try (FileInputStream in = FileUtils.openInputStream((File)directory);){
            Assert.fail();
        }
        catch (IOException iOException) {
            // empty catch block
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void openOutputStream_noParent(boolean createFile) throws Exception {
        File file = new File("test.txt");
        Assert.assertNull((Object)file.getParentFile());
        try {
            if (createFile) {
                TestUtils.createLineBasedFile(file, new String[]{"Hello"});
            }
            try (FileOutputStream out = FileUtils.openOutputStream((File)file);){
                out.write(0);
            }
            Assert.assertTrue((boolean)file.exists());
        }
        finally {
            if (!file.delete()) {
                file.deleteOnExit();
            }
        }
    }

    @Test
    public void test_openOutputStream_noParentCreateFile() throws Exception {
        this.openOutputStream_noParent(true);
    }

    @Test
    public void test_openOutputStream_noParentNoFile() throws Exception {
        this.openOutputStream_noParent(false);
    }

    @Test
    public void test_openOutputStream_exists() throws Exception {
        File file = new File(this.getTestDirectory(), "test.txt");
        TestUtils.createLineBasedFile(file, new String[]{"Hello"});
        try (FileOutputStream out = FileUtils.openOutputStream((File)file);){
            out.write(0);
        }
        Assert.assertTrue((boolean)file.exists());
    }

    @Test
    public void test_openOutputStream_existsButIsDirectory() throws Exception {
        File directory = new File(this.getTestDirectory(), "subdir");
        directory.mkdirs();
        try (FileOutputStream out = FileUtils.openOutputStream((File)directory);){
            Assert.fail();
        }
        catch (IOException iOException) {
            // empty catch block
        }
    }

    @Test
    public void test_openOutputStream_notExists() throws Exception {
        File file = new File(this.getTestDirectory(), "a/test.txt");
        try (FileOutputStream out = FileUtils.openOutputStream((File)file);){
            out.write(0);
        }
        Assert.assertTrue((boolean)file.exists());
    }

    @Test
    public void test_openOutputStream_notExistsCannotCreate() throws Exception {
        String longStr = "abcdevwxyzabcdevwxyzabcdevwxyzabcdevwxyzabcdevwxyzabcdevwxyzabcdevwxyzabcdevwxyzabcdevwxyzabcdevwxyzabcdevwxyzabcdevwxyzabcdevwxyzabcdevwxyzabcdevwxyzabcdevwxyzabcdevwxyzabcdevwxyzabcdevwxyzabcdevwxyzabcdevwxyzabcdevwxyzabcdevwxyzabcdevwxyzabcdevwxyzabcdevwxyzabcdevwxyzabcdevwxyzabcdevwxyzabcdevwxyz";
        File file = new File(this.getTestDirectory(), "a/abcdevwxyzabcdevwxyzabcdevwxyzabcdevwxyzabcdevwxyzabcdevwxyzabcdevwxyzabcdevwxyzabcdevwxyzabcdevwxyzabcdevwxyzabcdevwxyzabcdevwxyzabcdevwxyzabcdevwxyzabcdevwxyzabcdevwxyzabcdevwxyzabcdevwxyzabcdevwxyzabcdevwxyzabcdevwxyzabcdevwxyzabcdevwxyzabcdevwxyzabcdevwxyzabcdevwxyzabcdevwxyzabcdevwxyzabcdevwxyz/test.txt");
        try (FileOutputStream out = FileUtils.openOutputStream((File)file);){
            Assert.fail();
        }
        catch (IOException iOException) {
            // empty catch block
        }
    }

    @Test
    public void testByteCountToDisplaySizeBigInteger() {
        BigInteger b1023 = BigInteger.valueOf(1023L);
        BigInteger b1025 = BigInteger.valueOf(1025L);
        BigInteger KB1 = BigInteger.valueOf(1024L);
        BigInteger MB1 = KB1.multiply(KB1);
        BigInteger GB1 = MB1.multiply(KB1);
        BigInteger GB2 = GB1.add(GB1);
        BigInteger TB1 = GB1.multiply(KB1);
        BigInteger PB1 = TB1.multiply(KB1);
        BigInteger EB1 = PB1.multiply(KB1);
        Assert.assertEquals((Object)FileUtils.byteCountToDisplaySize((BigInteger)BigInteger.ZERO), (Object)"0 bytes");
        Assert.assertEquals((Object)FileUtils.byteCountToDisplaySize((BigInteger)BigInteger.ONE), (Object)"1 bytes");
        Assert.assertEquals((Object)FileUtils.byteCountToDisplaySize((BigInteger)b1023), (Object)"1023 bytes");
        Assert.assertEquals((Object)FileUtils.byteCountToDisplaySize((BigInteger)KB1), (Object)"1 KB");
        Assert.assertEquals((Object)FileUtils.byteCountToDisplaySize((BigInteger)b1025), (Object)"1 KB");
        Assert.assertEquals((Object)FileUtils.byteCountToDisplaySize((BigInteger)MB1.subtract(BigInteger.ONE)), (Object)"1023 KB");
        Assert.assertEquals((Object)FileUtils.byteCountToDisplaySize((BigInteger)MB1), (Object)"1 MB");
        Assert.assertEquals((Object)FileUtils.byteCountToDisplaySize((BigInteger)MB1.add(BigInteger.ONE)), (Object)"1 MB");
        Assert.assertEquals((Object)FileUtils.byteCountToDisplaySize((BigInteger)GB1.subtract(BigInteger.ONE)), (Object)"1023 MB");
        Assert.assertEquals((Object)FileUtils.byteCountToDisplaySize((BigInteger)GB1), (Object)"1 GB");
        Assert.assertEquals((Object)FileUtils.byteCountToDisplaySize((BigInteger)GB1.add(BigInteger.ONE)), (Object)"1 GB");
        Assert.assertEquals((Object)FileUtils.byteCountToDisplaySize((BigInteger)GB2), (Object)"2 GB");
        Assert.assertEquals((Object)FileUtils.byteCountToDisplaySize((BigInteger)GB2.subtract(BigInteger.ONE)), (Object)"1 GB");
        Assert.assertEquals((Object)FileUtils.byteCountToDisplaySize((BigInteger)TB1), (Object)"1 TB");
        Assert.assertEquals((Object)FileUtils.byteCountToDisplaySize((BigInteger)PB1), (Object)"1 PB");
        Assert.assertEquals((Object)FileUtils.byteCountToDisplaySize((BigInteger)EB1), (Object)"1 EB");
        Assert.assertEquals((Object)FileUtils.byteCountToDisplaySize((long)Long.MAX_VALUE), (Object)"7 EB");
        Assert.assertEquals((Object)FileUtils.byteCountToDisplaySize((BigInteger)BigInteger.valueOf(65535L)), (Object)"63 KB");
        Assert.assertEquals((Object)FileUtils.byteCountToDisplaySize((BigInteger)BigInteger.valueOf(32767L)), (Object)"31 KB");
        Assert.assertEquals((Object)FileUtils.byteCountToDisplaySize((BigInteger)BigInteger.valueOf(Integer.MAX_VALUE)), (Object)"1 GB");
    }

    @Test
    public void testByteCountToDisplaySizeLong() {
        Assert.assertEquals((Object)FileUtils.byteCountToDisplaySize((long)0L), (Object)"0 bytes");
        Assert.assertEquals((Object)FileUtils.byteCountToDisplaySize((long)1L), (Object)"1 bytes");
        Assert.assertEquals((Object)FileUtils.byteCountToDisplaySize((long)1023L), (Object)"1023 bytes");
        Assert.assertEquals((Object)FileUtils.byteCountToDisplaySize((long)1024L), (Object)"1 KB");
        Assert.assertEquals((Object)FileUtils.byteCountToDisplaySize((long)1025L), (Object)"1 KB");
        Assert.assertEquals((Object)FileUtils.byteCountToDisplaySize((long)1047552L), (Object)"1023 KB");
        Assert.assertEquals((Object)FileUtils.byteCountToDisplaySize((long)0x100000L), (Object)"1 MB");
        Assert.assertEquals((Object)FileUtils.byteCountToDisplaySize((long)0x100400L), (Object)"1 MB");
        Assert.assertEquals((Object)FileUtils.byteCountToDisplaySize((long)0x3FF00000L), (Object)"1023 MB");
        Assert.assertEquals((Object)FileUtils.byteCountToDisplaySize((long)0x40000000L), (Object)"1 GB");
        Assert.assertEquals((Object)FileUtils.byteCountToDisplaySize((long)0x40100000L), (Object)"1 GB");
        Assert.assertEquals((Object)FileUtils.byteCountToDisplaySize((long)0x80000000L), (Object)"2 GB");
        Assert.assertEquals((Object)FileUtils.byteCountToDisplaySize((long)Integer.MAX_VALUE), (Object)"1 GB");
        Assert.assertEquals((Object)FileUtils.byteCountToDisplaySize((long)0x10000000000L), (Object)"1 TB");
        Assert.assertEquals((Object)FileUtils.byteCountToDisplaySize((long)0x4000000000000L), (Object)"1 PB");
        Assert.assertEquals((Object)FileUtils.byteCountToDisplaySize((long)0x1000000000000000L), (Object)"1 EB");
        Assert.assertEquals((Object)FileUtils.byteCountToDisplaySize((long)Long.MAX_VALUE), (Object)"7 EB");
        Assert.assertEquals((Object)FileUtils.byteCountToDisplaySize((long)65535L), (Object)"63 KB");
        Assert.assertEquals((Object)FileUtils.byteCountToDisplaySize((long)32767L), (Object)"31 KB");
        Assert.assertEquals((Object)FileUtils.byteCountToDisplaySize((long)Integer.MAX_VALUE), (Object)"1 GB");
    }

    @Test
    public void testToFile1() throws Exception {
        URL url = new URL("file", null, "a/b/c/file.txt");
        File file = FileUtils.toFile((URL)url);
        Assert.assertTrue((boolean)file.toString().contains("file.txt"));
    }

    @Test
    public void testToFile2() throws Exception {
        URL url = new URL("file", null, "a/b/c/file%20n%61me%2520.tx%74");
        File file = FileUtils.toFile((URL)url);
        Assert.assertTrue((boolean)file.toString().contains("file name%20.txt"));
    }

    @Test
    public void testToFile3() throws Exception {
        Assert.assertEquals(null, (Object)FileUtils.toFile(null));
        Assert.assertEquals(null, (Object)FileUtils.toFile((URL)new URL("http://jakarta.apache.org")));
    }

    @Test
    public void testToFile4() throws Exception {
        URL url = new URL("file", null, "a/b/c/file%%20%me.txt%");
        File file = FileUtils.toFile((URL)url);
        Assert.assertTrue((boolean)file.toString().contains("file% %me.txt%"));
    }

    @Test
    public void testToFile5() throws Exception {
        URL url = new URL("file", null, "both%20are%20100%20%25%20true");
        File file = FileUtils.toFile((URL)url);
        Assert.assertEquals((Object)"both are 100 % true", (Object)file.toString());
    }

    @Test
    public void testToFileUtf8() throws Exception {
        URL url = new URL("file", null, "/home/%C3%A4%C3%B6%C3%BC%C3%9F");
        File file = FileUtils.toFile((URL)url);
        Assert.assertTrue((boolean)file.toString().contains("\u00e4\u00f6\u00fc\u00df"));
    }

    @Test
    public void testDecodeUrl() {
        Assert.assertEquals((Object)"", (Object)FileUtils.decodeUrl((String)""));
        Assert.assertEquals((Object)"foo", (Object)FileUtils.decodeUrl((String)"foo"));
        Assert.assertEquals((Object)"+", (Object)FileUtils.decodeUrl((String)"+"));
        Assert.assertEquals((Object)"% ", (Object)FileUtils.decodeUrl((String)"%25%20"));
        Assert.assertEquals((Object)"%20", (Object)FileUtils.decodeUrl((String)"%2520"));
        Assert.assertEquals((Object)"jar:file:/C:/dir/sub dir/1.0/foo-1.0.jar!/org/Bar.class", (Object)FileUtils.decodeUrl((String)"jar:file:/C:/dir/sub%20dir/1.0/foo-1.0.jar!/org/Bar.class"));
    }

    @Test
    public void testDecodeUrlLenient() {
        Assert.assertEquals((Object)" ", (Object)FileUtils.decodeUrl((String)" "));
        Assert.assertEquals((Object)"\u00e4\u00f6\u00fc\u00df", (Object)FileUtils.decodeUrl((String)"\u00e4\u00f6\u00fc\u00df"));
        Assert.assertEquals((Object)"%", (Object)FileUtils.decodeUrl((String)"%"));
        Assert.assertEquals((Object)"% ", (Object)FileUtils.decodeUrl((String)"%%20"));
        Assert.assertEquals((Object)"%2", (Object)FileUtils.decodeUrl((String)"%2"));
        Assert.assertEquals((Object)"%2G", (Object)FileUtils.decodeUrl((String)"%2G"));
    }

    @Test
    public void testDecodeUrlNullSafe() {
        Assert.assertNull((Object)FileUtils.decodeUrl(null));
    }

    @Test
    public void testDecodeUrlEncodingUtf8() {
        Assert.assertEquals((Object)"\u00e4\u00f6\u00fc\u00df", (Object)FileUtils.decodeUrl((String)"%C3%A4%C3%B6%C3%BC%C3%9F"));
    }

    @Test
    public void testToFiles1() throws Exception {
        URL[] urls = new URL[]{new URL("file", null, "file1.txt"), new URL("file", null, "file2.txt")};
        File[] files = FileUtils.toFiles((URL[])urls);
        Assert.assertEquals((long)urls.length, (long)files.length);
        Assert.assertEquals((String)("File: " + files[0]), (Object)true, (Object)files[0].toString().contains("file1.txt"));
        Assert.assertEquals((String)("File: " + files[1]), (Object)true, (Object)files[1].toString().contains("file2.txt"));
    }

    @Test
    public void testToFiles2() throws Exception {
        URL[] urls = new URL[]{new URL("file", null, "file1.txt"), null};
        File[] files = FileUtils.toFiles((URL[])urls);
        Assert.assertEquals((long)urls.length, (long)files.length);
        Assert.assertEquals((String)("File: " + files[0]), (Object)true, (Object)files[0].toString().contains("file1.txt"));
        Assert.assertEquals((String)("File: " + files[1]), null, (Object)files[1]);
    }

    @Test
    public void testToFiles3() throws Exception {
        URL[] urls = null;
        File[] files = FileUtils.toFiles(urls);
        Assert.assertEquals((long)0L, (long)files.length);
    }

    @Test
    public void testToFiles3a() throws Exception {
        URL[] urls = new URL[]{};
        File[] files = FileUtils.toFiles((URL[])urls);
        Assert.assertEquals((long)0L, (long)files.length);
    }

    @Test
    public void testToFiles4() throws Exception {
        URL[] urls = new URL[]{new URL("file", null, "file1.txt"), new URL("http", "jakarta.apache.org", "file1.txt")};
        try {
            FileUtils.toFiles((URL[])urls);
            Assert.fail();
        }
        catch (IllegalArgumentException illegalArgumentException) {
            // empty catch block
        }
    }

    @Test
    public void testToURLs1() throws Exception {
        File[] files = new File[]{new File(this.getTestDirectory(), "file1.txt"), new File(this.getTestDirectory(), "file2.txt"), new File(this.getTestDirectory(), "test file.txt")};
        URL[] urls = FileUtils.toURLs((File[])files);
        Assert.assertEquals((long)files.length, (long)urls.length);
        Assert.assertTrue((boolean)urls[0].toExternalForm().startsWith("file:"));
        Assert.assertTrue((boolean)urls[0].toExternalForm().contains("file1.txt"));
        Assert.assertTrue((boolean)urls[1].toExternalForm().startsWith("file:"));
        Assert.assertTrue((boolean)urls[1].toExternalForm().contains("file2.txt"));
        Assert.assertTrue((boolean)urls[2].toExternalForm().startsWith("file:"));
        Assert.assertTrue((boolean)urls[2].toExternalForm().contains("test%20file.txt"));
    }

    @Test
    public void testToURLs3a() throws Exception {
        File[] files = new File[]{};
        URL[] urls = FileUtils.toURLs((File[])files);
        Assert.assertEquals((long)0L, (long)urls.length);
    }

    @Test
    public void testContentEquals() throws Exception {
        File file = new File(this.getTestDirectory(), this.getName());
        File file2 = new File(this.getTestDirectory(), this.getName() + "2");
        Assert.assertTrue((boolean)FileUtils.contentEquals((File)file, (File)file));
        Assert.assertTrue((boolean)FileUtils.contentEquals((File)file, (File)file2));
        Assert.assertTrue((boolean)FileUtils.contentEquals((File)file2, (File)file2));
        Assert.assertTrue((boolean)FileUtils.contentEquals((File)file2, (File)file));
        try {
            FileUtils.contentEquals((File)this.getTestDirectory(), (File)this.getTestDirectory());
            Assert.fail((String)"Comparing directories should fail with an IOException");
        }
        catch (IOException iOException) {
            // empty catch block
        }
        File objFile1 = new File(this.getTestDirectory(), this.getName() + ".object");
        objFile1.deleteOnExit();
        FileUtils.copyURLToFile((URL)this.getClass().getResource("/java/lang/Object.class"), (File)objFile1);
        File objFile1b = new File(this.getTestDirectory(), this.getName() + ".object2");
        objFile1.deleteOnExit();
        FileUtils.copyURLToFile((URL)this.getClass().getResource("/java/lang/Object.class"), (File)objFile1b);
        File objFile2 = new File(this.getTestDirectory(), this.getName() + ".collection");
        objFile2.deleteOnExit();
        FileUtils.copyURLToFile((URL)this.getClass().getResource("/java/util/Collection.class"), (File)objFile2);
        Assert.assertFalse((boolean)FileUtils.contentEquals((File)objFile1, (File)objFile2));
        Assert.assertFalse((boolean)FileUtils.contentEquals((File)objFile1b, (File)objFile2));
        Assert.assertTrue((boolean)FileUtils.contentEquals((File)objFile1, (File)objFile1b));
        Assert.assertTrue((boolean)FileUtils.contentEquals((File)objFile1, (File)objFile1));
        Assert.assertTrue((boolean)FileUtils.contentEquals((File)objFile1b, (File)objFile1b));
        Assert.assertTrue((boolean)FileUtils.contentEquals((File)objFile2, (File)objFile2));
        file.createNewFile();
        file2.createNewFile();
        Assert.assertTrue((boolean)FileUtils.contentEquals((File)file, (File)file));
        Assert.assertTrue((boolean)FileUtils.contentEquals((File)file, (File)file2));
    }

    @Test
    public void testContentEqualsIgnoreEOL() throws Exception {
        File file1 = new File(this.getTestDirectory(), this.getName());
        File file2 = new File(this.getTestDirectory(), this.getName() + "2");
        Assert.assertTrue((boolean)FileUtils.contentEqualsIgnoreEOL((File)file1, (File)file1, null));
        Assert.assertTrue((boolean)FileUtils.contentEqualsIgnoreEOL((File)file1, (File)file2, null));
        Assert.assertTrue((boolean)FileUtils.contentEqualsIgnoreEOL((File)file2, (File)file2, null));
        Assert.assertTrue((boolean)FileUtils.contentEqualsIgnoreEOL((File)file2, (File)file1, null));
        try {
            FileUtils.contentEqualsIgnoreEOL((File)this.getTestDirectory(), (File)this.getTestDirectory(), null);
            Assert.fail((String)"Comparing directories should fail with an IOException");
        }
        catch (IOException iOException) {
            // empty catch block
        }
        File tfile1 = new File(this.getTestDirectory(), this.getName() + ".txt1");
        tfile1.deleteOnExit();
        FileUtils.write((File)tfile1, (CharSequence)"123\r");
        File tfile2 = new File(this.getTestDirectory(), this.getName() + ".txt2");
        tfile1.deleteOnExit();
        FileUtils.write((File)tfile2, (CharSequence)"123\n");
        File tfile3 = new File(this.getTestDirectory(), this.getName() + ".collection");
        tfile3.deleteOnExit();
        FileUtils.write((File)tfile3, (CharSequence)"123\r\n2");
        Assert.assertTrue((boolean)FileUtils.contentEqualsIgnoreEOL((File)tfile1, (File)tfile1, null));
        Assert.assertTrue((boolean)FileUtils.contentEqualsIgnoreEOL((File)tfile2, (File)tfile2, null));
        Assert.assertTrue((boolean)FileUtils.contentEqualsIgnoreEOL((File)tfile3, (File)tfile3, null));
        Assert.assertTrue((boolean)FileUtils.contentEqualsIgnoreEOL((File)tfile1, (File)tfile2, null));
        Assert.assertFalse((boolean)FileUtils.contentEqualsIgnoreEOL((File)tfile1, (File)tfile3, null));
        Assert.assertFalse((boolean)FileUtils.contentEqualsIgnoreEOL((File)tfile2, (File)tfile3, null));
        URL urlCR = this.getClass().getResource("FileUtilsTestDataCR.dat");
        Assert.assertNotNull((Object)urlCR);
        File cr = new File(urlCR.getPath());
        Assert.assertTrue((boolean)cr.exists());
        URL urlCRLF = this.getClass().getResource("FileUtilsTestDataCRLF.dat");
        Assert.assertNotNull((Object)urlCRLF);
        File crlf = new File(urlCRLF.getPath());
        Assert.assertTrue((boolean)crlf.exists());
        URL urlLF = this.getClass().getResource("FileUtilsTestDataLF.dat");
        Assert.assertNotNull((Object)urlLF);
        File lf = new File(urlLF.getPath());
        Assert.assertTrue((boolean)lf.exists());
        Assert.assertTrue((boolean)FileUtils.contentEqualsIgnoreEOL((File)cr, (File)cr, null));
        Assert.assertTrue((boolean)FileUtils.contentEqualsIgnoreEOL((File)crlf, (File)crlf, null));
        Assert.assertTrue((boolean)FileUtils.contentEqualsIgnoreEOL((File)lf, (File)lf, null));
        Assert.assertTrue((boolean)FileUtils.contentEqualsIgnoreEOL((File)cr, (File)crlf, null));
        Assert.assertTrue((boolean)FileUtils.contentEqualsIgnoreEOL((File)cr, (File)lf, null));
        Assert.assertTrue((boolean)FileUtils.contentEqualsIgnoreEOL((File)crlf, (File)lf, null));
        Assert.assertTrue((boolean)FileUtils.contentEquals((File)cr, (File)cr));
        Assert.assertTrue((boolean)FileUtils.contentEquals((File)crlf, (File)crlf));
        Assert.assertTrue((boolean)FileUtils.contentEquals((File)lf, (File)lf));
        Assert.assertFalse((boolean)FileUtils.contentEquals((File)cr, (File)crlf));
        Assert.assertFalse((boolean)FileUtils.contentEquals((File)cr, (File)lf));
        Assert.assertFalse((boolean)FileUtils.contentEquals((File)crlf, (File)lf));
        file1.createNewFile();
        file2.createNewFile();
        Assert.assertTrue((boolean)FileUtils.contentEqualsIgnoreEOL((File)file1, (File)file1, null));
        Assert.assertTrue((boolean)FileUtils.contentEqualsIgnoreEOL((File)file1, (File)file2, null));
    }

    @Test
    public void testCopyURLToFile() throws Exception {
        File file = new File(this.getTestDirectory(), this.getName());
        file.deleteOnExit();
        String resourceName = "/java/lang/Object.class";
        FileUtils.copyURLToFile((URL)this.getClass().getResource("/java/lang/Object.class"), (File)file);
        try (FileInputStream fis = new FileInputStream(file);){
            Assert.assertTrue((String)"Content is not equal.", (boolean)IOUtils.contentEquals((InputStream)this.getClass().getResourceAsStream("/java/lang/Object.class"), (InputStream)fis));
        }
    }

    @Test
    public void testCopyURLToFileWithTimeout() throws Exception {
        File file = new File(this.getTestDirectory(), "testCopyURLToFileWithTimeout");
        file.deleteOnExit();
        String resourceName = "/java/lang/Object.class";
        FileUtils.copyURLToFile((URL)this.getClass().getResource("/java/lang/Object.class"), (File)file, (int)500, (int)500);
        try (FileInputStream fis = new FileInputStream(file);){
            Assert.assertTrue((String)"Content is not equal.", (boolean)IOUtils.contentEquals((InputStream)this.getClass().getResourceAsStream("/java/lang/Object.class"), (InputStream)fis));
        }
    }

    @Test
    public void testForceMkdir() throws Exception {
        FileUtils.forceMkdir((File)this.getTestDirectory());
        File testFile = new File(this.getTestDirectory(), this.getName());
        testFile.deleteOnExit();
        testFile.createNewFile();
        Assert.assertTrue((String)"Test file does not exist.", (boolean)testFile.exists());
        try {
            FileUtils.forceMkdir((File)testFile);
            Assert.fail((String)"Exception expected.");
        }
        catch (IOException iOException) {
            // empty catch block
        }
        testFile.delete();
        FileUtils.forceMkdir((File)testFile);
        Assert.assertTrue((String)"Directory was not created.", (boolean)testFile.exists());
    }

    @Test
    public void testForceMkdirParent() throws Exception {
        Assert.assertTrue((boolean)this.getTestDirectory().exists());
        File testParentDir = new File(this.getTestDirectory(), "testForceMkdirParent");
        testParentDir.delete();
        Assert.assertFalse((boolean)testParentDir.exists());
        File testFile = new File(testParentDir, "test.txt");
        Assert.assertFalse((boolean)testParentDir.exists());
        Assert.assertFalse((boolean)testFile.exists());
        FileUtils.forceMkdirParent((File)testFile);
        Assert.assertTrue((boolean)testParentDir.exists());
        Assert.assertFalse((boolean)testFile.exists());
        FileUtils.forceMkdirParent((File)testFile);
        Assert.assertTrue((boolean)testParentDir.exists());
        Assert.assertFalse((boolean)testFile.exists());
    }

    @Test
    public void testSizeOfDirectory() throws Exception {
        File file = new File(this.getTestDirectory(), this.getName());
        try {
            FileUtils.sizeOfDirectory((File)file);
            Assert.fail((String)"Exception expected.");
        }
        catch (IllegalArgumentException illegalArgumentException) {
            // empty catch block
        }
        file.createNewFile();
        try {
            FileUtils.sizeOfDirectory((File)file);
            Assert.fail((String)"Exception expected.");
        }
        catch (IllegalArgumentException illegalArgumentException) {
            // empty catch block
        }
        file.delete();
        file.mkdir();
        this.createCircularSymLink(file);
        Assert.assertEquals((String)"Unexpected directory size", (long)0L, (long)FileUtils.sizeOfDirectory((File)file));
    }

    private void createCircularSymLink(File file) throws IOException {
        if (!FilenameUtils.isSystemWindows()) {
            Runtime.getRuntime().exec("ln -s " + file + "/.. " + file + "/cycle");
        } else {
            try {
                Runtime.getRuntime().exec("mklink /D " + file + "/cycle" + file + "/.. ");
            }
            catch (IOException iOException) {
                // empty catch block
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    public void testSizeOfDirectoryAsBigInteger() throws Exception {
        File file = new File(this.getTestDirectory(), this.getName());
        try {
            FileUtils.sizeOfDirectoryAsBigInteger((File)file);
            Assert.fail((String)"Exception expected.");
        }
        catch (IllegalArgumentException illegalArgumentException) {
            // empty catch block
        }
        file.createNewFile();
        file.deleteOnExit();
        try {
            FileUtils.sizeOfDirectoryAsBigInteger((File)file);
            Assert.fail((String)"Exception expected.");
        }
        catch (IllegalArgumentException illegalArgumentException) {
            // empty catch block
        }
        file.delete();
        file.mkdir();
        this.createCircularSymLink(file);
        Assert.assertEquals((String)"Unexpected directory size", (Object)TEST_DIRECTORY_SIZE_BI, (Object)FileUtils.sizeOfDirectoryAsBigInteger((File)file));
        file.delete();
        file.mkdir();
        File nonEmptyFile = new File(file, "nonEmptyFile" + System.nanoTime());
        if (!nonEmptyFile.getParentFile().exists()) {
            throw new IOException("Cannot create file " + nonEmptyFile + " as the parent directory does not exist");
        }
        BufferedOutputStream output = new BufferedOutputStream(new FileOutputStream(nonEmptyFile));
        try {
            TestUtils.generateTestData(output, TEST_DIRECTORY_SIZE_GT_ZERO_BI.longValue());
        }
        finally {
            IOUtils.closeQuietly((OutputStream)output);
        }
        nonEmptyFile.deleteOnExit();
        Assert.assertEquals((String)"Unexpected directory size", (Object)TEST_DIRECTORY_SIZE_GT_ZERO_BI, (Object)FileUtils.sizeOfDirectoryAsBigInteger((File)file));
        nonEmptyFile.delete();
        file.delete();
    }

    @Test
    public void testCompareSizeOf() {
        File start = new File("src/test/java");
        long sizeLong1 = FileUtils.sizeOf((File)start);
        BigInteger sizeBig = FileUtils.sizeOfAsBigInteger((File)start);
        long sizeLong2 = FileUtils.sizeOf((File)start);
        Assert.assertEquals((String)"Size should not change", (long)sizeLong1, (long)sizeLong2);
        Assert.assertEquals((String)"longSize should equal BigSize", (long)sizeLong1, (long)sizeBig.longValue());
    }

    @Test
    public void testSizeOf() throws Exception {
        File file = new File(this.getTestDirectory(), this.getName());
        try {
            FileUtils.sizeOf(null);
            Assert.fail((String)"Exception expected.");
        }
        catch (NullPointerException nullPointerException) {
            // empty catch block
        }
        try {
            FileUtils.sizeOf((File)file);
            Assert.fail((String)"Exception expected.");
        }
        catch (IllegalArgumentException illegalArgumentException) {
            // empty catch block
        }
        file.createNewFile();
        file.deleteOnExit();
        Assert.assertEquals((long)0L, (long)FileUtils.sizeOf((File)file));
        file.delete();
        Assert.assertEquals((String)"Unexpected files size", (long)this.testFile1Size, (long)FileUtils.sizeOf((File)this.testFile1));
        Assert.assertEquals((String)"Unexpected directory size", (long)0L, (long)FileUtils.sizeOf((File)this.getTestDirectory()));
    }

    @Test
    public void testSizeOfAsBigInteger() throws Exception {
        File file = new File(this.getTestDirectory(), this.getName());
        try {
            FileUtils.sizeOfAsBigInteger(null);
            Assert.fail((String)"Exception expected.");
        }
        catch (NullPointerException nullPointerException) {
            // empty catch block
        }
        try {
            FileUtils.sizeOfAsBigInteger((File)file);
            Assert.fail((String)"Exception expected.");
        }
        catch (IllegalArgumentException illegalArgumentException) {
            // empty catch block
        }
        file.createNewFile();
        file.deleteOnExit();
        Assert.assertEquals((Object)BigInteger.ZERO, (Object)FileUtils.sizeOfAsBigInteger((File)file));
        file.delete();
        Assert.assertEquals((String)"Unexpected files size", (Object)BigInteger.valueOf(this.testFile1Size), (Object)FileUtils.sizeOfAsBigInteger((File)this.testFile1));
        Assert.assertEquals((String)"Unexpected directory size", (Object)TEST_DIRECTORY_SIZE_BI, (Object)FileUtils.sizeOfAsBigInteger((File)this.getTestDirectory()));
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    public void testIsFileNewerOlder() throws Exception {
        String message;
        File reference = new File(this.getTestDirectory(), "FileUtils-reference.txt");
        File oldFile = new File(this.getTestDirectory(), "FileUtils-old.txt");
        File newFile = new File(this.getTestDirectory(), "FileUtils-new.txt");
        File invalidFile = new File(this.getTestDirectory(), "FileUtils-invalid-file.txt");
        if (!oldFile.getParentFile().exists()) {
            throw new IOException("Cannot create file " + oldFile + " as the parent directory does not exist");
        }
        BufferedOutputStream output1 = new BufferedOutputStream(new FileOutputStream(oldFile));
        try {
            TestUtils.generateTestData(output1, 0L);
        }
        finally {
            IOUtils.closeQuietly((OutputStream)output1);
        }
        do {
            try {
                TestUtils.sleep(1000L);
            }
            catch (InterruptedException interruptedException) {
                // empty catch block
            }
            if (!reference.getParentFile().exists()) {
                throw new IOException("Cannot create file " + reference + " as the parent directory does not exist");
            }
            BufferedOutputStream output = new BufferedOutputStream(new FileOutputStream(reference));
            try {
                TestUtils.generateTestData(output, 0L);
            }
            finally {
                IOUtils.closeQuietly((OutputStream)output);
            }
        } while (oldFile.lastModified() == reference.lastModified());
        Date date = new Date();
        long now = date.getTime();
        do {
            try {
                TestUtils.sleep(1000L);
            }
            catch (InterruptedException interruptedException) {
                // empty catch block
            }
            if (!newFile.getParentFile().exists()) {
                throw new IOException("Cannot create file " + newFile + " as the parent directory does not exist");
            }
            BufferedOutputStream output = new BufferedOutputStream(new FileOutputStream(newFile));
            try {
                TestUtils.generateTestData(output, 0L);
            }
            finally {
                IOUtils.closeQuietly((OutputStream)output);
            }
        } while (reference.lastModified() == newFile.lastModified());
        Assert.assertFalse((String)"Old File - Newer - File", (boolean)FileUtils.isFileNewer((File)oldFile, (File)reference));
        Assert.assertFalse((String)"Old File - Newer - Date", (boolean)FileUtils.isFileNewer((File)oldFile, (Date)date));
        Assert.assertFalse((String)"Old File - Newer - Mili", (boolean)FileUtils.isFileNewer((File)oldFile, (long)now));
        Assert.assertTrue((String)"New File - Newer - File", (boolean)FileUtils.isFileNewer((File)newFile, (File)reference));
        Assert.assertTrue((String)"New File - Newer - Date", (boolean)FileUtils.isFileNewer((File)newFile, (Date)date));
        Assert.assertTrue((String)"New File - Newer - Mili", (boolean)FileUtils.isFileNewer((File)newFile, (long)now));
        Assert.assertFalse((String)"Invalid - Newer - File", (boolean)FileUtils.isFileNewer((File)invalidFile, (File)reference));
        String invalidFileName = invalidFile.getName();
        try {
            FileUtils.isFileNewer((File)newFile, (File)invalidFile);
            Assert.fail((String)"Should have cause IllegalArgumentException");
        }
        catch (IllegalArgumentException iae) {
            message = iae.getMessage();
            Assert.assertTrue((String)("Message should contain: " + invalidFileName + " but was: " + message), (boolean)message.contains(invalidFileName));
        }
        Assert.assertTrue((String)"Old File - Older - File", (boolean)FileUtils.isFileOlder((File)oldFile, (File)reference));
        Assert.assertTrue((String)"Old File - Older - Date", (boolean)FileUtils.isFileOlder((File)oldFile, (Date)date));
        Assert.assertTrue((String)"Old File - Older - Mili", (boolean)FileUtils.isFileOlder((File)oldFile, (long)now));
        Assert.assertFalse((String)"New File - Older - File", (boolean)FileUtils.isFileOlder((File)newFile, (File)reference));
        Assert.assertFalse((String)"New File - Older - Date", (boolean)FileUtils.isFileOlder((File)newFile, (Date)date));
        Assert.assertFalse((String)"New File - Older - Mili", (boolean)FileUtils.isFileOlder((File)newFile, (long)now));
        Assert.assertFalse((String)"Invalid - Older - File", (boolean)FileUtils.isFileOlder((File)invalidFile, (File)reference));
        try {
            FileUtils.isFileOlder((File)newFile, (File)invalidFile);
            Assert.fail((String)"Should have cause IllegalArgumentException");
        }
        catch (IllegalArgumentException iae) {
            message = iae.getMessage();
            Assert.assertTrue((String)("Message should contain: " + invalidFileName + " but was: " + message), (boolean)message.contains(invalidFileName));
        }
        try {
            FileUtils.isFileNewer(null, (long)now);
            Assert.fail((String)"Newer Null, expected IllegalArgumentExcepion");
        }
        catch (IllegalArgumentException illegalArgumentException) {
            // empty catch block
        }
        try {
            FileUtils.isFileNewer((File)oldFile, (File)null);
            Assert.fail((String)"Newer Null reference, expected IllegalArgumentExcepion");
        }
        catch (IllegalArgumentException illegalArgumentException) {
            // empty catch block
        }
        try {
            FileUtils.isFileNewer((File)oldFile, (File)invalidFile);
            Assert.fail((String)"Newer invalid reference, expected IllegalArgumentExcepion");
        }
        catch (IllegalArgumentException illegalArgumentException) {
            // empty catch block
        }
        try {
            FileUtils.isFileNewer((File)oldFile, (Date)null);
            Assert.fail((String)"Newer Null date, expected IllegalArgumentExcepion");
        }
        catch (IllegalArgumentException illegalArgumentException) {
            // empty catch block
        }
        try {
            FileUtils.isFileOlder(null, (long)now);
            Assert.fail((String)"Older Null, expected IllegalArgumentExcepion");
        }
        catch (IllegalArgumentException illegalArgumentException) {
            // empty catch block
        }
        try {
            FileUtils.isFileOlder((File)oldFile, (File)null);
            Assert.fail((String)"Older Null reference, expected IllegalArgumentExcepion");
        }
        catch (IllegalArgumentException illegalArgumentException) {
            // empty catch block
        }
        try {
            FileUtils.isFileOlder((File)oldFile, (File)invalidFile);
            Assert.fail((String)"Older invalid reference, expected IllegalArgumentExcepion");
        }
        catch (IllegalArgumentException illegalArgumentException) {
            // empty catch block
        }
        try {
            FileUtils.isFileOlder((File)oldFile, (Date)null);
            Assert.fail((String)"Older Null date, expected IllegalArgumentExcepion");
        }
        catch (IllegalArgumentException illegalArgumentException) {
            // empty catch block
        }
    }

    @Test
    public void testCopyFile1() throws Exception {
        File destination = new File(this.getTestDirectory(), "copy1.txt");
        FileUtils.copyFile((File)this.testFile1, (File)destination);
        Assert.assertTrue((String)"Check Exist", (boolean)destination.exists());
        Assert.assertEquals((String)"Check Full copy", (long)this.testFile1Size, (long)destination.length());
    }

    @Test
    public void testCopyFileToOutputStream() throws Exception {
        ByteArrayOutputStream destination = new ByteArrayOutputStream();
        FileUtils.copyFile((File)this.testFile1, (OutputStream)destination);
        Assert.assertEquals((String)"Check Full copy size", (long)this.testFile1Size, (long)destination.size());
        byte[] expected = FileUtils.readFileToByteArray((File)this.testFile1);
        Assert.assertArrayEquals((String)"Check Full copy", (byte[])expected, (byte[])destination.toByteArray());
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    @Ignore
    public void testCopyFileLarge() throws Exception {
        File largeFile = new File(this.getTestDirectory(), "large.txt");
        File destination = new File(this.getTestDirectory(), "copylarge.txt");
        System.out.println("START:   " + new Date());
        if (!largeFile.getParentFile().exists()) {
            throw new IOException("Cannot create file " + largeFile + " as the parent directory does not exist");
        }
        BufferedOutputStream output = new BufferedOutputStream(new FileOutputStream(largeFile));
        try {
            TestUtils.generateTestData(output, 0x40000000L);
        }
        finally {
            IOUtils.closeQuietly((OutputStream)output);
        }
        System.out.println("CREATED: " + new Date());
        FileUtils.copyFile((File)largeFile, (File)destination);
        System.out.println("COPIED:  " + new Date());
        Assert.assertTrue((String)"Check Exist", (boolean)destination.exists());
        Assert.assertEquals((String)"Check Full copy", (long)largeFile.length(), (long)destination.length());
    }

    @Test
    public void testCopyFile2() throws Exception {
        File destination = new File(this.getTestDirectory(), "copy2.txt");
        FileUtils.copyFile((File)this.testFile1, (File)destination);
        Assert.assertTrue((String)"Check Exist", (boolean)destination.exists());
        Assert.assertEquals((String)"Check Full copy", (long)this.testFile2Size, (long)destination.length());
    }

    @Test
    public void testCopyToSelf() throws Exception {
        File destination = new File(this.getTestDirectory(), "copy3.txt");
        FileUtils.copyFile((File)this.testFile1, (File)destination);
        try {
            FileUtils.copyFile((File)destination, (File)destination);
            Assert.fail((String)"file copy to self should not be possible");
        }
        catch (IOException iOException) {
            // empty catch block
        }
    }

    @Test
    public void testCopyFile2WithoutFileDatePreservation() throws Exception {
        File destination = new File(this.getTestDirectory(), "copy2.txt");
        FileUtils.copyFile((File)this.testFile1, (File)destination, (boolean)false);
        Assert.assertTrue((String)"Check Exist", (boolean)destination.exists());
        Assert.assertEquals((String)"Check Full copy", (long)this.testFile2Size, (long)destination.length());
    }

    @Test
    public void testCopyDirectoryToDirectory_NonExistingDest() throws Exception {
        if (!this.testFile1.getParentFile().exists()) {
            throw new IOException("Cannot create file " + this.testFile1 + " as the parent directory does not exist");
        }
        BufferedOutputStream output1 = new BufferedOutputStream(new FileOutputStream(this.testFile1));
        try {
            TestUtils.generateTestData(output1, 1234L);
        }
        finally {
            IOUtils.closeQuietly((OutputStream)output1);
        }
        if (!this.testFile2.getParentFile().exists()) {
            throw new IOException("Cannot create file " + this.testFile2 + " as the parent directory does not exist");
        }
        BufferedOutputStream output = new BufferedOutputStream(new FileOutputStream(this.testFile2));
        try {
            TestUtils.generateTestData(output, 4321L);
        }
        finally {
            IOUtils.closeQuietly((OutputStream)output);
        }
        File srcDir = this.getTestDirectory();
        File subDir = new File(srcDir, "sub");
        subDir.mkdir();
        File subFile = new File(subDir, "A.txt");
        FileUtils.writeStringToFile((File)subFile, (String)"HELLO WORLD", (String)"UTF8");
        File destDir = new File(System.getProperty("java.io.tmpdir"), "tmp-FileUtilsTestCase");
        FileUtils.deleteDirectory((File)destDir);
        File actualDestDir = new File(destDir, srcDir.getName());
        FileUtils.copyDirectoryToDirectory((File)srcDir, (File)destDir);
        Assert.assertTrue((String)"Check exists", (boolean)destDir.exists());
        Assert.assertTrue((String)"Check exists", (boolean)actualDestDir.exists());
        long srcSize = FileUtils.sizeOfDirectory((File)srcDir);
        Assert.assertTrue((String)"Size > 0", (srcSize > 0L ? 1 : 0) != 0);
        Assert.assertEquals((String)"Check size", (long)srcSize, (long)FileUtils.sizeOfDirectory((File)actualDestDir));
        Assert.assertTrue((boolean)new File(actualDestDir, "sub/A.txt").exists());
        FileUtils.deleteDirectory((File)destDir);
    }

    @Test
    public void testCopyDirectoryToNonExistingDest() throws Exception {
        if (!this.testFile1.getParentFile().exists()) {
            throw new IOException("Cannot create file " + this.testFile1 + " as the parent directory does not exist");
        }
        BufferedOutputStream output1 = new BufferedOutputStream(new FileOutputStream(this.testFile1));
        try {
            TestUtils.generateTestData(output1, 1234L);
        }
        finally {
            IOUtils.closeQuietly((OutputStream)output1);
        }
        if (!this.testFile2.getParentFile().exists()) {
            throw new IOException("Cannot create file " + this.testFile2 + " as the parent directory does not exist");
        }
        BufferedOutputStream output = new BufferedOutputStream(new FileOutputStream(this.testFile2));
        try {
            TestUtils.generateTestData(output, 4321L);
        }
        finally {
            IOUtils.closeQuietly((OutputStream)output);
        }
        File srcDir = this.getTestDirectory();
        File subDir = new File(srcDir, "sub");
        subDir.mkdir();
        File subFile = new File(subDir, "A.txt");
        FileUtils.writeStringToFile((File)subFile, (String)"HELLO WORLD", (String)"UTF8");
        File destDir = new File(System.getProperty("java.io.tmpdir"), "tmp-FileUtilsTestCase");
        FileUtils.deleteDirectory((File)destDir);
        FileUtils.copyDirectory((File)srcDir, (File)destDir);
        Assert.assertTrue((String)"Check exists", (boolean)destDir.exists());
        long sizeOfSrcDirectory = FileUtils.sizeOfDirectory((File)srcDir);
        Assert.assertTrue((String)"Size > 0", (sizeOfSrcDirectory > 0L ? 1 : 0) != 0);
        Assert.assertEquals((String)"Check size", (long)sizeOfSrcDirectory, (long)FileUtils.sizeOfDirectory((File)destDir));
        Assert.assertTrue((boolean)new File(destDir, "sub/A.txt").exists());
        FileUtils.deleteDirectory((File)destDir);
    }

    @Test
    public void testCopyDirectoryToExistingDest() throws Exception {
        if (!this.testFile1.getParentFile().exists()) {
            throw new IOException("Cannot create file " + this.testFile1 + " as the parent directory does not exist");
        }
        BufferedOutputStream output1 = new BufferedOutputStream(new FileOutputStream(this.testFile1));
        try {
            TestUtils.generateTestData(output1, 1234L);
        }
        finally {
            IOUtils.closeQuietly((OutputStream)output1);
        }
        if (!this.testFile2.getParentFile().exists()) {
            throw new IOException("Cannot create file " + this.testFile2 + " as the parent directory does not exist");
        }
        BufferedOutputStream output = new BufferedOutputStream(new FileOutputStream(this.testFile2));
        try {
            TestUtils.generateTestData(output, 4321L);
        }
        finally {
            IOUtils.closeQuietly((OutputStream)output);
        }
        File srcDir = this.getTestDirectory();
        File subDir = new File(srcDir, "sub");
        subDir.mkdir();
        File subFile = new File(subDir, "A.txt");
        FileUtils.writeStringToFile((File)subFile, (String)"HELLO WORLD", (String)"UTF8");
        File destDir = new File(System.getProperty("java.io.tmpdir"), "tmp-FileUtilsTestCase");
        FileUtils.deleteDirectory((File)destDir);
        destDir.mkdirs();
        FileUtils.copyDirectory((File)srcDir, (File)destDir);
        long srcSize = FileUtils.sizeOfDirectory((File)srcDir);
        Assert.assertTrue((String)"Size > 0", (srcSize > 0L ? 1 : 0) != 0);
        Assert.assertEquals((long)srcSize, (long)FileUtils.sizeOfDirectory((File)destDir));
        Assert.assertTrue((boolean)new File(destDir, "sub/A.txt").exists());
    }

    @Test
    public void testCopyDirectoryFiltered() throws Exception {
        File grandParentDir = new File(this.getTestDirectory(), "grandparent");
        File parentDir = new File(grandParentDir, "parent");
        File childDir = new File(parentDir, "child");
        this.createFilesForTestCopyDirectory(grandParentDir, parentDir, childDir);
        NameFileFilter filter = new NameFileFilter(new String[]{"parent", "child", "file3.txt"});
        File destDir = new File(this.getTestDirectory(), "copydest");
        FileUtils.copyDirectory((File)grandParentDir, (File)destDir, (FileFilter)filter);
        List<File> files = LIST_WALKER.list(destDir);
        Assert.assertEquals((long)3L, (long)files.size());
        Assert.assertEquals((Object)"parent", (Object)files.get(0).getName());
        Assert.assertEquals((Object)"child", (Object)files.get(1).getName());
        Assert.assertEquals((Object)"file3.txt", (Object)files.get(2).getName());
    }

    @Test
    public void testCopyDirectoryPreserveDates() throws Exception {
        File source = new File(this.getTestDirectory(), "source");
        File sourceDirectory = new File(source, "directory");
        File sourceFile = new File(sourceDirectory, "hello.txt");
        source.mkdirs();
        sourceDirectory.mkdir();
        FileUtils.writeStringToFile((File)sourceFile, (String)"HELLO WORLD", (String)"UTF8");
        sourceFile.setLastModified(1000000002000L);
        sourceDirectory.setLastModified(1000000001000L);
        source.setLastModified(1000000000000L);
        File target = new File(this.getTestDirectory(), "target");
        File targetDirectory = new File(target, "directory");
        File targetFile = new File(targetDirectory, "hello.txt");
        FileUtils.copyDirectory((File)source, (File)target, (boolean)false);
        Assert.assertTrue((1000000000000L != target.lastModified() ? 1 : 0) != 0);
        Assert.assertTrue((1000000001000L != targetDirectory.lastModified() ? 1 : 0) != 0);
        Assert.assertTrue((1000000002000L != targetFile.lastModified() ? 1 : 0) != 0);
        FileUtils.deleteDirectory((File)target);
        FileUtils.copyDirectory((File)source, (File)target, (boolean)true);
        Assert.assertEquals((long)1000000000000L, (long)target.lastModified());
        Assert.assertEquals((long)1000000001000L, (long)targetDirectory.lastModified());
        Assert.assertEquals((long)1000000002000L, (long)targetFile.lastModified());
        FileUtils.deleteDirectory((File)target);
        target.mkdirs();
        FileUtils.copyDirectory((File)source, (File)target, (boolean)true);
        Assert.assertEquals((long)1000000000000L, (long)target.lastModified());
        Assert.assertEquals((long)1000000001000L, (long)targetDirectory.lastModified());
        Assert.assertEquals((long)1000000002000L, (long)targetFile.lastModified());
        FileUtils.deleteDirectory((File)target);
        targetDirectory.mkdirs();
        FileUtils.copyDirectory((File)source, (File)target, (boolean)true);
        Assert.assertEquals((long)1000000000000L, (long)target.lastModified());
        Assert.assertEquals((long)1000000001000L, (long)targetDirectory.lastModified());
        Assert.assertEquals((long)1000000002000L, (long)targetFile.lastModified());
        FileUtils.deleteDirectory((File)target);
    }

    @Test
    public void testCopyDirectoryToChild() throws Exception {
        File grandParentDir = new File(this.getTestDirectory(), "grandparent");
        File parentDir = new File(grandParentDir, "parent");
        File childDir = new File(parentDir, "child");
        this.createFilesForTestCopyDirectory(grandParentDir, parentDir, childDir);
        long expectedCount = LIST_WALKER.list(grandParentDir).size() + LIST_WALKER.list(parentDir).size();
        long expectedSize = FileUtils.sizeOfDirectory((File)grandParentDir) + FileUtils.sizeOfDirectory((File)parentDir);
        FileUtils.copyDirectory((File)parentDir, (File)childDir);
        Assert.assertEquals((long)expectedCount, (long)LIST_WALKER.list(grandParentDir).size());
        Assert.assertEquals((long)expectedSize, (long)FileUtils.sizeOfDirectory((File)grandParentDir));
        Assert.assertTrue((String)"Count > 0", (expectedCount > 0L ? 1 : 0) != 0);
        Assert.assertTrue((String)"Size > 0", (expectedSize > 0L ? 1 : 0) != 0);
    }

    @Test
    public void testCopyDirectoryToGrandChild() throws Exception {
        File grandParentDir = new File(this.getTestDirectory(), "grandparent");
        File parentDir = new File(grandParentDir, "parent");
        File childDir = new File(parentDir, "child");
        this.createFilesForTestCopyDirectory(grandParentDir, parentDir, childDir);
        long expectedCount = LIST_WALKER.list(grandParentDir).size() * 2;
        long expectedSize = FileUtils.sizeOfDirectory((File)grandParentDir) * 2L;
        FileUtils.copyDirectory((File)grandParentDir, (File)childDir);
        Assert.assertEquals((long)expectedCount, (long)LIST_WALKER.list(grandParentDir).size());
        Assert.assertEquals((long)expectedSize, (long)FileUtils.sizeOfDirectory((File)grandParentDir));
        Assert.assertTrue((String)"Size > 0", (expectedSize > 0L ? 1 : 0) != 0);
    }

    @Test
    public void testCopyDirectoryToItself() throws Exception {
        File dir = new File(this.getTestDirectory(), "itself");
        dir.mkdirs();
        FileUtils.copyDirectoryToDirectory((File)dir, (File)dir);
        Assert.assertEquals((long)1L, (long)LIST_WALKER.list(dir).size());
    }

    private void createFilesForTestCopyDirectory(File grandParentDir, File parentDir, File childDir) throws Exception {
        File childDir2 = new File(parentDir, "child2");
        File grandChildDir = new File(childDir, "grandChild");
        File grandChild2Dir = new File(childDir2, "grandChild2");
        File file1 = new File(grandParentDir, "file1.txt");
        File file2 = new File(parentDir, "file2.txt");
        File file3 = new File(childDir, "file3.txt");
        File file4 = new File(childDir2, "file4.txt");
        File file5 = new File(grandChildDir, "file5.txt");
        File file6 = new File(grandChild2Dir, "file6.txt");
        FileUtils.deleteDirectory((File)grandParentDir);
        grandChildDir.mkdirs();
        grandChild2Dir.mkdirs();
        FileUtils.writeStringToFile((File)file1, (String)"File 1 in grandparent", (String)"UTF8");
        FileUtils.writeStringToFile((File)file2, (String)"File 2 in parent", (String)"UTF8");
        FileUtils.writeStringToFile((File)file3, (String)"File 3 in child", (String)"UTF8");
        FileUtils.writeStringToFile((File)file4, (String)"File 4 in child2", (String)"UTF8");
        FileUtils.writeStringToFile((File)file5, (String)"File 5 in grandChild", (String)"UTF8");
        FileUtils.writeStringToFile((File)file6, (String)"File 6 in grandChild2", (String)"UTF8");
    }

    @Test
    public void testCopyDirectoryErrors() throws Exception {
        try {
            FileUtils.copyDirectory(null, null);
            Assert.fail();
        }
        catch (NullPointerException nullPointerException) {
            // empty catch block
        }
        try {
            FileUtils.copyDirectory((File)new File("a"), null);
            Assert.fail();
        }
        catch (NullPointerException nullPointerException) {
            // empty catch block
        }
        try {
            FileUtils.copyDirectory(null, (File)new File("a"));
            Assert.fail();
        }
        catch (NullPointerException nullPointerException) {
            // empty catch block
        }
        try {
            FileUtils.copyDirectory((File)new File("doesnt-exist"), (File)new File("a"));
            Assert.fail();
        }
        catch (IOException iOException) {
            // empty catch block
        }
        try {
            FileUtils.copyDirectory((File)this.testFile1, (File)new File("a"));
            Assert.fail();
        }
        catch (IOException iOException) {
            // empty catch block
        }
        try {
            FileUtils.copyDirectory((File)this.getTestDirectory(), (File)this.testFile1);
            Assert.fail();
        }
        catch (IOException iOException) {
            // empty catch block
        }
        try {
            FileUtils.copyDirectory((File)this.getTestDirectory(), (File)this.getTestDirectory());
            Assert.fail();
        }
        catch (IOException iOException) {
            // empty catch block
        }
    }

    @Test
    public void testCopyToDirectoryWithFile() throws IOException {
        File directory = new File(this.getTestDirectory(), "subdir");
        if (!directory.exists()) {
            directory.mkdirs();
        }
        File destination = new File(directory, this.testFile1.getName());
        FileUtils.copyToDirectory((File)this.testFile1, (File)directory);
        Assert.assertTrue((String)"Check Exists", (boolean)destination.exists());
        Assert.assertEquals((String)"Check Full Copy", (long)this.testFile1Size, (long)destination.length());
    }

    @Test(expected=NullPointerException.class)
    public void testCopyToDirectoryWithFileSourceIsNull() throws IOException {
        FileUtils.copyToDirectory((File)null, (File)this.getTestDirectory());
    }

    @Test(expected=IOException.class)
    public void testCopyToDirectoryWithFileSourceDoesNotExist() throws IOException {
        FileUtils.copyToDirectory((File)new File(this.getTestDirectory(), "doesNotExists"), (File)this.getTestDirectory());
    }

    @Test
    public void testCopyToDirectoryWithDirectory() throws IOException {
        File inputDirectory;
        File destDirectory = new File(this.getTestDirectory(), "destination");
        if (!destDirectory.exists()) {
            destDirectory.mkdirs();
        }
        if (!(inputDirectory = new File(this.getTestDirectory(), "input")).exists()) {
            inputDirectory.mkdirs();
        }
        File outputDirDestination = new File(destDirectory, inputDirectory.getName());
        FileUtils.copyToDirectory((File)this.testFile1, (File)inputDirectory);
        File destFile1 = new File(outputDirDestination, this.testFile1.getName());
        FileUtils.copyToDirectory((File)this.testFile2, (File)inputDirectory);
        File destFile2 = new File(outputDirDestination, this.testFile2.getName());
        FileUtils.copyToDirectory((File)inputDirectory, (File)destDirectory);
        Assert.assertTrue((String)"Check Exists", (boolean)outputDirDestination.exists());
        Assert.assertTrue((String)"Check Directory", (boolean)outputDirDestination.isDirectory());
        Assert.assertTrue((String)"Check Exists", (boolean)destFile1.exists());
        Assert.assertEquals((String)"Check Full Copy", (long)this.testFile1Size, (long)destFile1.length());
        Assert.assertTrue((String)"Check Exists", (boolean)destFile2.exists());
        Assert.assertEquals((String)"Check Full Copy", (long)this.testFile2Size, (long)destFile2.length());
    }

    @Test
    public void testCopyToDirectoryWithIterable() throws IOException {
        File directory = new File(this.getTestDirectory(), "subdir");
        if (!directory.exists()) {
            directory.mkdirs();
        }
        ArrayList<File> input = new ArrayList<File>();
        input.add(this.testFile1);
        input.add(this.testFile2);
        File destFile1 = new File(directory, this.testFile1.getName());
        File destFile2 = new File(directory, this.testFile2.getName());
        FileUtils.copyToDirectory(input, (File)directory);
        Assert.assertTrue((String)"Check Exists", (boolean)destFile1.exists());
        Assert.assertEquals((String)"Check Full Copy", (long)this.testFile1Size, (long)destFile1.length());
        Assert.assertTrue((String)"Check Exists", (boolean)destFile2.exists());
        Assert.assertEquals((String)"Check Full Copy", (long)this.testFile2Size, (long)destFile2.length());
    }

    @Test(expected=NullPointerException.class)
    public void testCopyToDirectoryWithIterableSourceIsNull() throws IOException {
        FileUtils.copyToDirectory((Iterable)null, (File)this.getTestDirectory());
    }

    @Test(expected=IOException.class)
    public void testCopyToDirectoryWithIterableSourceDoesNotExist() throws IOException {
        FileUtils.copyToDirectory(Collections.singleton(new File(this.getTestDirectory(), "doesNotExists")), (File)this.getTestDirectory());
    }

    @Test
    public void testForceDeleteAFile1() throws Exception {
        File destination = new File(this.getTestDirectory(), "copy1.txt");
        destination.createNewFile();
        Assert.assertTrue((String)"Copy1.txt doesn't exist to delete", (boolean)destination.exists());
        FileUtils.forceDelete((File)destination);
        Assert.assertTrue((String)"Check No Exist", (!destination.exists() ? 1 : 0) != 0);
    }

    @Test
    public void testForceDeleteAFile2() throws Exception {
        File destination = new File(this.getTestDirectory(), "copy2.txt");
        destination.createNewFile();
        Assert.assertTrue((String)"Copy2.txt doesn't exist to delete", (boolean)destination.exists());
        FileUtils.forceDelete((File)destination);
        Assert.assertTrue((String)"Check No Exist", (!destination.exists() ? 1 : 0) != 0);
    }

    @Test
    public void testForceDeleteAFile3() throws Exception {
        File destination = new File(this.getTestDirectory(), "no_such_file");
        Assert.assertTrue((String)"Check No Exist", (!destination.exists() ? 1 : 0) != 0);
        try {
            FileUtils.forceDelete((File)destination);
            Assert.fail((String)"Should generate FileNotFoundException");
        }
        catch (FileNotFoundException fileNotFoundException) {
            // empty catch block
        }
    }

    @Test
    public void testCopyFile1ToDir() throws Exception {
        File directory = new File(this.getTestDirectory(), "subdir");
        if (!directory.exists()) {
            directory.mkdirs();
        }
        File destination = new File(directory, this.testFile1.getName());
        FileUtils.copyFileToDirectory((File)this.testFile1, (File)directory);
        Assert.assertTrue((String)"Check Exist", (boolean)destination.exists());
        Assert.assertEquals((String)"Check Full copy", (long)this.testFile1Size, (long)destination.length());
        try {
            FileUtils.copyFileToDirectory((File)destination, (File)directory);
            Assert.fail((String)"Should not be able to copy a file into the same directory as itself");
        }
        catch (IOException iOException) {
            // empty catch block
        }
    }

    @Test
    public void testCopyFile2ToDir() throws Exception {
        File directory = new File(this.getTestDirectory(), "subdir");
        if (!directory.exists()) {
            directory.mkdirs();
        }
        File destination = new File(directory, this.testFile1.getName());
        FileUtils.copyFileToDirectory((File)this.testFile1, (File)directory);
        Assert.assertTrue((String)"Check Exist", (boolean)destination.exists());
        Assert.assertEquals((String)"Check Full copy", (long)this.testFile2Size, (long)destination.length());
    }

    @Test
    public void testForceDeleteDir() throws Exception {
        File testDirectory = this.getTestDirectory();
        Assert.assertTrue((String)"TestDirectory must exist", (boolean)testDirectory.exists());
        FileUtils.forceDelete((File)testDirectory);
        Assert.assertFalse((String)"TestDirectory must not exist", (boolean)testDirectory.exists());
    }

    @Test
    public void testFileUtils() throws Exception {
        File file1 = new File(this.getTestDirectory(), "test.txt");
        String filename = file1.getAbsolutePath();
        try (FileOutputStream out = new FileOutputStream(file1);){
            ((OutputStream)out).write("This is a test".getBytes("UTF-8"));
        }
        File file2 = new File(this.getTestDirectory(), "test2.txt");
        FileUtils.writeStringToFile((File)file2, (String)filename, (String)"UTF-8");
        Assert.assertTrue((boolean)file2.exists());
        Assert.assertTrue((file2.length() > 0L ? 1 : 0) != 0);
        String file2contents = FileUtils.readFileToString((File)file2, (String)"UTF-8");
        Assert.assertTrue((String)"Second file's contents correct", (boolean)filename.equals(file2contents));
        Assert.assertTrue((boolean)file2.delete());
        String contents = FileUtils.readFileToString((File)new File(filename), (String)"UTF-8");
        Assert.assertEquals((String)"FileUtils.fileRead()", (Object)"This is a test", (Object)contents);
    }

    @Test
    public void testTouch() throws IOException {
        File file = new File(this.getTestDirectory(), "touch.txt");
        if (file.exists()) {
            file.delete();
        }
        Assert.assertTrue((String)"Bad test: test file still exists", (!file.exists() ? 1 : 0) != 0);
        FileUtils.touch((File)file);
        Assert.assertTrue((String)"FileUtils.touch() created file", (boolean)file.exists());
        FileOutputStream out = new FileOutputStream(file);
        Assert.assertEquals((String)"Created empty file.", (long)0L, (long)file.length());
        out.write(0);
        out.close();
        Assert.assertEquals((String)"Wrote one byte to file", (long)1L, (long)file.length());
        long y2k = new GregorianCalendar(2000, 0, 1).getTime().getTime();
        boolean res = file.setLastModified(y2k);
        Assert.assertEquals((String)"Bad test: set lastModified failed", (Object)true, (Object)res);
        Assert.assertEquals((String)"Bad test: set lastModified set incorrect value", (long)y2k, (long)file.lastModified());
        long now = System.currentTimeMillis();
        FileUtils.touch((File)file);
        Assert.assertEquals((String)"FileUtils.touch() didn't empty the file.", (long)1L, (long)file.length());
        Assert.assertEquals((String)"FileUtils.touch() changed lastModified", (Object)false, (Object)(y2k == file.lastModified() ? 1 : 0));
        Assert.assertEquals((String)"FileUtils.touch() changed lastModified to more than now-3s", (Object)true, (Object)(file.lastModified() >= now - 3000L ? 1 : 0));
        Assert.assertEquals((String)"FileUtils.touch() changed lastModified to less than now+3s", (Object)true, (Object)(file.lastModified() <= now + 3000L ? 1 : 0));
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    public void testListFiles() throws Exception {
        File srcDir = this.getTestDirectory();
        File subDir = new File(srcDir, "list_test");
        subDir.mkdir();
        File subDir2 = new File(subDir, "subdir");
        subDir2.mkdir();
        String[] fileNames = new String[]{"a.txt", "b.txt", "c.txt", "d.txt", "e.txt", "f.txt"};
        int[] fileSizes = new int[]{123, 234, 345, 456, 678, 789};
        for (int i = 0; i < fileNames.length; ++i) {
            File theFile = new File(subDir, fileNames[i]);
            if (!theFile.getParentFile().exists()) {
                throw new IOException("Cannot create file " + theFile + " as the parent directory does not exist");
            }
            BufferedOutputStream output = new BufferedOutputStream(new FileOutputStream(theFile));
            try {
                TestUtils.generateTestData(output, fileSizes[i]);
                continue;
            }
            finally {
                IOUtils.closeQuietly((OutputStream)output);
            }
        }
        Collection files = FileUtils.listFiles((File)subDir, (IOFileFilter)new WildcardFileFilter("*.*"), (IOFileFilter)new WildcardFileFilter("*"));
        int count = files.size();
        Object[] fileObjs = files.toArray();
        Assert.assertEquals((long)fileNames.length, (long)files.size());
        HashMap<String, String> foundFileNames = new HashMap<String, String>();
        for (int i = 0; i < count; ++i) {
            boolean found = false;
            for (int j = 0; !found && j < fileNames.length; ++j) {
                if (!fileNames[j].equals(((File)fileObjs[i]).getName())) continue;
                foundFileNames.put(fileNames[j], fileNames[j]);
                found = true;
            }
        }
        Assert.assertEquals((long)foundFileNames.size(), (long)fileNames.length);
        subDir.delete();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    public void testListFilesWithDirs() throws IOException {
        File srcDir = this.getTestDirectory();
        File subDir1 = new File(srcDir, "subdir");
        subDir1.mkdir();
        File subDir2 = new File(subDir1, "subdir2");
        subDir2.mkdir();
        File someFile = new File(subDir2, "a.txt");
        if (!someFile.getParentFile().exists()) {
            throw new IOException("Cannot create file " + someFile + " as the parent directory does not exist");
        }
        BufferedOutputStream output = new BufferedOutputStream(new FileOutputStream(someFile));
        try {
            TestUtils.generateTestData(output, 100L);
        }
        finally {
            IOUtils.closeQuietly((OutputStream)output);
        }
        File subDir3 = new File(subDir2, "subdir3");
        subDir3.mkdir();
        Collection files = FileUtils.listFilesAndDirs((File)subDir1, (IOFileFilter)new WildcardFileFilter("*.*"), (IOFileFilter)new WildcardFileFilter("*"));
        Assert.assertEquals((long)4L, (long)files.size());
        Assert.assertTrue((String)"Should contain the directory.", (boolean)files.contains(subDir1));
        Assert.assertTrue((String)"Should contain the directory.", (boolean)files.contains(subDir2));
        Assert.assertTrue((String)"Should contain the file.", (boolean)files.contains(someFile));
        Assert.assertTrue((String)"Should contain the directory.", (boolean)files.contains(subDir3));
        subDir1.delete();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    public void testIterateFiles() throws Exception {
        File srcDir = this.getTestDirectory();
        File subDir = new File(srcDir, "list_test");
        subDir.mkdir();
        String[] fileNames = new String[]{"a.txt", "b.txt", "c.txt", "d.txt", "e.txt", "f.txt"};
        int[] fileSizes = new int[]{123, 234, 345, 456, 678, 789};
        for (int i = 0; i < fileNames.length; ++i) {
            File theFile = new File(subDir, fileNames[i]);
            if (!theFile.getParentFile().exists()) {
                throw new IOException("Cannot create file " + theFile + " as the parent directory does not exist");
            }
            BufferedOutputStream output = new BufferedOutputStream(new FileOutputStream(theFile));
            try {
                TestUtils.generateTestData(output, fileSizes[i]);
                continue;
            }
            finally {
                IOUtils.closeQuietly((OutputStream)output);
            }
        }
        Iterator files = FileUtils.iterateFiles((File)subDir, (IOFileFilter)new WildcardFileFilter("*.*"), (IOFileFilter)new WildcardFileFilter("*"));
        HashMap<String, String> foundFileNames = new HashMap<String, String>();
        while (files.hasNext()) {
            boolean found = false;
            String fileName = ((File)files.next()).getName();
            for (int j = 0; !found && j < fileNames.length; ++j) {
                if (!fileNames[j].equals(fileName)) continue;
                foundFileNames.put(fileNames[j], fileNames[j]);
                found = true;
            }
        }
        Assert.assertEquals((long)foundFileNames.size(), (long)fileNames.length);
        subDir.delete();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    public void testIterateFilesAndDirs() throws IOException {
        File srcDir = this.getTestDirectory();
        File subDir1 = new File(srcDir, "subdir");
        subDir1.mkdir();
        File subDir2 = new File(subDir1, "subdir2");
        subDir2.mkdir();
        File someFile = new File(subDir2, "a.txt");
        if (!someFile.getParentFile().exists()) {
            throw new IOException("Cannot create file " + someFile + " as the parent directory does not exist");
        }
        BufferedOutputStream output = new BufferedOutputStream(new FileOutputStream(someFile));
        try {
            TestUtils.generateTestData(output, 100L);
        }
        finally {
            IOUtils.closeQuietly((OutputStream)output);
        }
        File subDir3 = new File(subDir2, "subdir3");
        subDir3.mkdir();
        List<File> filesAndDirs = Arrays.asList(subDir1, subDir2, someFile, subDir3);
        int filesCount = 0;
        Iterator files = FileUtils.iterateFilesAndDirs((File)subDir1, (IOFileFilter)new WildcardFileFilter("*.*"), (IOFileFilter)new WildcardFileFilter("*"));
        while (files.hasNext()) {
            ++filesCount;
            File file = (File)files.next();
            Assert.assertTrue((String)"Should contain the directory/file", (boolean)filesAndDirs.contains(file));
        }
        Assert.assertEquals((long)filesCount, (long)filesAndDirs.size());
    }

    @Test
    public void testReadFileToStringWithDefaultEncoding() throws Exception {
        File file = new File(this.getTestDirectory(), "read.obj");
        FileOutputStream out = new FileOutputStream(file);
        byte[] text = "Hello /u1234".getBytes();
        out.write(text);
        out.close();
        String data = FileUtils.readFileToString((File)file);
        Assert.assertEquals((Object)"Hello /u1234", (Object)data);
    }

    @Test
    public void testReadFileToStringWithEncoding() throws Exception {
        File file = new File(this.getTestDirectory(), "read.obj");
        FileOutputStream out = new FileOutputStream(file);
        byte[] text = "Hello /u1234".getBytes("UTF8");
        out.write(text);
        out.close();
        String data = FileUtils.readFileToString((File)file, (String)"UTF8");
        Assert.assertEquals((Object)"Hello /u1234", (Object)data);
    }

    @Test
    public void testReadFileToByteArray() throws Exception {
        File file = new File(this.getTestDirectory(), "read.txt");
        FileOutputStream out = new FileOutputStream(file);
        out.write(11);
        out.write(21);
        out.write(31);
        out.close();
        byte[] data = FileUtils.readFileToByteArray((File)file);
        Assert.assertEquals((long)3L, (long)data.length);
        Assert.assertEquals((long)11L, (long)data[0]);
        Assert.assertEquals((long)21L, (long)data[1]);
        Assert.assertEquals((long)31L, (long)data[2]);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    public void testReadLines() throws Exception {
        File file = TestUtils.newFile(this.getTestDirectory(), "lines.txt");
        try {
            String[] data = new String[]{"hello", "/u1234", "", "this is", "some text"};
            TestUtils.createLineBasedFile(file, data);
            List lines = FileUtils.readLines((File)file, (String)"UTF-8");
            Assert.assertEquals(Arrays.asList(data), (Object)lines);
        }
        finally {
            TestUtils.deleteFile(file);
        }
    }

    @Test
    public void testWriteStringToFile1() throws Exception {
        File file = new File(this.getTestDirectory(), "write.txt");
        FileUtils.writeStringToFile((File)file, (String)"Hello /u1234", (String)"UTF8");
        byte[] text = "Hello /u1234".getBytes("UTF8");
        TestUtils.assertEqualContent(text, file);
    }

    @Test
    public void testWriteStringToFile2() throws Exception {
        File file = new File(this.getTestDirectory(), "write.txt");
        FileUtils.writeStringToFile((File)file, (String)"Hello /u1234", (String)null);
        byte[] text = "Hello /u1234".getBytes();
        TestUtils.assertEqualContent(text, file);
    }

    @Test
    public void testWriteStringToFile3() throws Exception {
        File file = new File(this.getTestDirectory(), "write.txt");
        FileUtils.writeStringToFile((File)file, (String)"Hello /u1234", (Charset)null);
        byte[] text = "Hello /u1234".getBytes();
        TestUtils.assertEqualContent(text, file);
    }

    @Test
    public void testWriteCharSequence1() throws Exception {
        File file = new File(this.getTestDirectory(), "write.txt");
        FileUtils.write((File)file, (CharSequence)"Hello /u1234", (String)"UTF8");
        byte[] text = "Hello /u1234".getBytes("UTF8");
        TestUtils.assertEqualContent(text, file);
    }

    @Test
    public void testWriteCharSequence2() throws Exception {
        File file = new File(this.getTestDirectory(), "write.txt");
        FileUtils.write((File)file, (CharSequence)"Hello /u1234", (String)null);
        byte[] text = "Hello /u1234".getBytes();
        TestUtils.assertEqualContent(text, file);
    }

    @Test
    public void testWriteByteArrayToFile() throws Exception {
        File file = new File(this.getTestDirectory(), "write.obj");
        byte[] data = new byte[]{11, 21, 31};
        FileUtils.writeByteArrayToFile((File)file, (byte[])data);
        TestUtils.assertEqualContent(data, file);
    }

    @Test
    public void testWriteByteArrayToFile_WithOffsetAndLength() throws Exception {
        File file = new File(this.getTestDirectory(), "write.obj");
        byte[] data = new byte[]{11, 21, 32, 41, 51};
        byte[] writtenData = new byte[3];
        System.arraycopy(data, 1, writtenData, 0, 3);
        FileUtils.writeByteArrayToFile((File)file, (byte[])data, (int)1, (int)3);
        TestUtils.assertEqualContent(writtenData, file);
    }

    @Test
    public void testWriteLines_4arg() throws Exception {
        Object[] data = new Object[]{"hello", new StringBuffer("world"), "", "this is", null, "some text"};
        List<Object> list = Arrays.asList(data);
        File file = TestUtils.newFile(this.getTestDirectory(), "lines.txt");
        FileUtils.writeLines((File)file, (String)"US-ASCII", list, (String)"*");
        String expected = "hello*world**this is**some text*";
        String actual = FileUtils.readFileToString((File)file, (String)"US-ASCII");
        Assert.assertEquals((Object)"hello*world**this is**some text*", (Object)actual);
    }

    @Test
    public void testWriteLines_4arg_Writer_nullData() throws Exception {
        File file = TestUtils.newFile(this.getTestDirectory(), "lines.txt");
        FileUtils.writeLines((File)file, (String)"US-ASCII", null, (String)"*");
        Assert.assertEquals((String)"Sizes differ", (long)0L, (long)file.length());
    }

    @Test
    public void testWriteLines_4arg_nullSeparator() throws Exception {
        Object[] data = new Object[]{"hello", new StringBuffer("world"), "", "this is", null, "some text"};
        List<Object> list = Arrays.asList(data);
        File file = TestUtils.newFile(this.getTestDirectory(), "lines.txt");
        FileUtils.writeLines((File)file, (String)"US-ASCII", list, null);
        String expected = "hello" + IOUtils.LINE_SEPARATOR + "world" + IOUtils.LINE_SEPARATOR + IOUtils.LINE_SEPARATOR + "this is" + IOUtils.LINE_SEPARATOR + IOUtils.LINE_SEPARATOR + "some text" + IOUtils.LINE_SEPARATOR;
        String actual = FileUtils.readFileToString((File)file, (String)"US-ASCII");
        Assert.assertEquals((Object)expected, (Object)actual);
    }

    @Test
    public void testWriteLines_3arg_nullSeparator() throws Exception {
        Object[] data = new Object[]{"hello", new StringBuffer("world"), "", "this is", null, "some text"};
        List<Object> list = Arrays.asList(data);
        File file = TestUtils.newFile(this.getTestDirectory(), "lines.txt");
        FileUtils.writeLines((File)file, (String)"US-ASCII", list);
        String expected = "hello" + IOUtils.LINE_SEPARATOR + "world" + IOUtils.LINE_SEPARATOR + IOUtils.LINE_SEPARATOR + "this is" + IOUtils.LINE_SEPARATOR + IOUtils.LINE_SEPARATOR + "some text" + IOUtils.LINE_SEPARATOR;
        String actual = FileUtils.readFileToString((File)file, (String)"US-ASCII");
        Assert.assertEquals((Object)expected, (Object)actual);
    }

    @Test
    public void testWriteLines_5argsWithAppendOptionTrue_ShouldNotDeletePreviousFileLines() throws Exception {
        File file = TestUtils.newFile(this.getTestDirectory(), "lines.txt");
        FileUtils.writeStringToFile((File)file, (String)"This line was there before you...");
        List<String> linesToAppend = Arrays.asList("my first line", "The second Line");
        FileUtils.writeLines((File)file, null, linesToAppend, null, (boolean)true);
        String expected = "This line was there before you...my first line" + IOUtils.LINE_SEPARATOR + "The second Line" + IOUtils.LINE_SEPARATOR;
        String actual = FileUtils.readFileToString((File)file);
        Assert.assertEquals((Object)expected, (Object)actual);
    }

    @Test
    public void testWriteLines_5argsWithAppendOptionFalse_ShouldDeletePreviousFileLines() throws Exception {
        File file = TestUtils.newFile(this.getTestDirectory(), "lines.txt");
        FileUtils.writeStringToFile((File)file, (String)"This line was there before you...");
        List<String> linesToAppend = Arrays.asList("my first line", "The second Line");
        FileUtils.writeLines((File)file, null, linesToAppend, null, (boolean)false);
        String expected = "my first line" + IOUtils.LINE_SEPARATOR + "The second Line" + IOUtils.LINE_SEPARATOR;
        String actual = FileUtils.readFileToString((File)file);
        Assert.assertEquals((Object)expected, (Object)actual);
    }

    @Test
    public void testWriteLines_4argsWithAppendOptionTrue_ShouldNotDeletePreviousFileLines() throws Exception {
        File file = TestUtils.newFile(this.getTestDirectory(), "lines.txt");
        FileUtils.writeStringToFile((File)file, (String)"This line was there before you...");
        List<String> linesToAppend = Arrays.asList("my first line", "The second Line");
        FileUtils.writeLines((File)file, linesToAppend, null, (boolean)true);
        String expected = "This line was there before you...my first line" + IOUtils.LINE_SEPARATOR + "The second Line" + IOUtils.LINE_SEPARATOR;
        String actual = FileUtils.readFileToString((File)file);
        Assert.assertEquals((Object)expected, (Object)actual);
    }

    @Test
    public void testWriteLines_4argsWithAppendOptionFalse_ShouldDeletePreviousFileLines() throws Exception {
        File file = TestUtils.newFile(this.getTestDirectory(), "lines.txt");
        FileUtils.writeStringToFile((File)file, (String)"This line was there before you...");
        List<String> linesToAppend = Arrays.asList("my first line", "The second Line");
        FileUtils.writeLines((File)file, linesToAppend, null, (boolean)false);
        String expected = "my first line" + IOUtils.LINE_SEPARATOR + "The second Line" + IOUtils.LINE_SEPARATOR;
        String actual = FileUtils.readFileToString((File)file);
        Assert.assertEquals((Object)expected, (Object)actual);
    }

    @Test
    public void testWriteLinesEncoding_WithAppendOptionTrue_ShouldNotDeletePreviousFileLines() throws Exception {
        File file = TestUtils.newFile(this.getTestDirectory(), "lines.txt");
        FileUtils.writeStringToFile((File)file, (String)"This line was there before you...");
        List<String> linesToAppend = Arrays.asList("my first line", "The second Line");
        FileUtils.writeLines((File)file, null, linesToAppend, (boolean)true);
        String expected = "This line was there before you...my first line" + IOUtils.LINE_SEPARATOR + "The second Line" + IOUtils.LINE_SEPARATOR;
        String actual = FileUtils.readFileToString((File)file);
        Assert.assertEquals((Object)expected, (Object)actual);
    }

    @Test
    public void testWriteLinesEncoding_WithAppendOptionFalse_ShouldDeletePreviousFileLines() throws Exception {
        File file = TestUtils.newFile(this.getTestDirectory(), "lines.txt");
        FileUtils.writeStringToFile((File)file, (String)"This line was there before you...");
        List<String> linesToAppend = Arrays.asList("my first line", "The second Line");
        FileUtils.writeLines((File)file, null, linesToAppend, (boolean)false);
        String expected = "my first line" + IOUtils.LINE_SEPARATOR + "The second Line" + IOUtils.LINE_SEPARATOR;
        String actual = FileUtils.readFileToString((File)file);
        Assert.assertEquals((Object)expected, (Object)actual);
    }

    @Test
    public void testWriteLines_3argsWithAppendOptionTrue_ShouldNotDeletePreviousFileLines() throws Exception {
        File file = TestUtils.newFile(this.getTestDirectory(), "lines.txt");
        FileUtils.writeStringToFile((File)file, (String)"This line was there before you...");
        List<String> linesToAppend = Arrays.asList("my first line", "The second Line");
        FileUtils.writeLines((File)file, linesToAppend, (boolean)true);
        String expected = "This line was there before you...my first line" + IOUtils.LINE_SEPARATOR + "The second Line" + IOUtils.LINE_SEPARATOR;
        String actual = FileUtils.readFileToString((File)file);
        Assert.assertEquals((Object)expected, (Object)actual);
    }

    @Test
    public void testWriteLines_3argsWithAppendOptionFalse_ShouldDeletePreviousFileLines() throws Exception {
        File file = TestUtils.newFile(this.getTestDirectory(), "lines.txt");
        FileUtils.writeStringToFile((File)file, (String)"This line was there before you...");
        List<String> linesToAppend = Arrays.asList("my first line", "The second Line");
        FileUtils.writeLines((File)file, linesToAppend, (boolean)false);
        String expected = "my first line" + IOUtils.LINE_SEPARATOR + "The second Line" + IOUtils.LINE_SEPARATOR;
        String actual = FileUtils.readFileToString((File)file);
        Assert.assertEquals((Object)expected, (Object)actual);
    }

    @Test
    public void testWriteStringToFileWithEncoding_WithAppendOptionTrue_ShouldNotDeletePreviousFileLines() throws Exception {
        File file = TestUtils.newFile(this.getTestDirectory(), "lines.txt");
        FileUtils.writeStringToFile((File)file, (String)"This line was there before you...");
        FileUtils.writeStringToFile((File)file, (String)"this is brand new data", (String)null, (boolean)true);
        String expected = "This line was there before you...this is brand new data";
        String actual = FileUtils.readFileToString((File)file);
        Assert.assertEquals((Object)"This line was there before you...this is brand new data", (Object)actual);
    }

    @Test
    public void testWriteStringToFileWithEncoding_WithAppendOptionFalse_ShouldDeletePreviousFileLines() throws Exception {
        File file = TestUtils.newFile(this.getTestDirectory(), "lines.txt");
        FileUtils.writeStringToFile((File)file, (String)"This line was there before you...");
        FileUtils.writeStringToFile((File)file, (String)"this is brand new data", (String)null, (boolean)false);
        String expected = "this is brand new data";
        String actual = FileUtils.readFileToString((File)file);
        Assert.assertEquals((Object)"this is brand new data", (Object)actual);
    }

    @Test
    public void testWriteStringToFile_WithAppendOptionTrue_ShouldNotDeletePreviousFileLines() throws Exception {
        File file = TestUtils.newFile(this.getTestDirectory(), "lines.txt");
        FileUtils.writeStringToFile((File)file, (String)"This line was there before you...");
        FileUtils.writeStringToFile((File)file, (String)"this is brand new data", (boolean)true);
        String expected = "This line was there before you...this is brand new data";
        String actual = FileUtils.readFileToString((File)file);
        Assert.assertEquals((Object)"This line was there before you...this is brand new data", (Object)actual);
    }

    @Test
    public void testWriteStringToFile_WithAppendOptionFalse_ShouldDeletePreviousFileLines() throws Exception {
        File file = TestUtils.newFile(this.getTestDirectory(), "lines.txt");
        FileUtils.writeStringToFile((File)file, (String)"This line was there before you...");
        FileUtils.writeStringToFile((File)file, (String)"this is brand new data", (boolean)false);
        String expected = "this is brand new data";
        String actual = FileUtils.readFileToString((File)file);
        Assert.assertEquals((Object)"this is brand new data", (Object)actual);
    }

    @Test
    public void testWriteWithEncoding_WithAppendOptionTrue_ShouldNotDeletePreviousFileLines() throws Exception {
        File file = TestUtils.newFile(this.getTestDirectory(), "lines.txt");
        FileUtils.writeStringToFile((File)file, (String)"This line was there before you...");
        FileUtils.write((File)file, (CharSequence)"this is brand new data", (String)null, (boolean)true);
        String expected = "This line was there before you...this is brand new data";
        String actual = FileUtils.readFileToString((File)file);
        Assert.assertEquals((Object)"This line was there before you...this is brand new data", (Object)actual);
    }

    @Test
    public void testWriteWithEncoding_WithAppendOptionFalse_ShouldDeletePreviousFileLines() throws Exception {
        File file = TestUtils.newFile(this.getTestDirectory(), "lines.txt");
        FileUtils.writeStringToFile((File)file, (String)"This line was there before you...");
        FileUtils.write((File)file, (CharSequence)"this is brand new data", (String)null, (boolean)false);
        String expected = "this is brand new data";
        String actual = FileUtils.readFileToString((File)file);
        Assert.assertEquals((Object)"this is brand new data", (Object)actual);
    }

    @Test
    public void testWrite_WithAppendOptionTrue_ShouldNotDeletePreviousFileLines() throws Exception {
        File file = TestUtils.newFile(this.getTestDirectory(), "lines.txt");
        FileUtils.writeStringToFile((File)file, (String)"This line was there before you...");
        FileUtils.write((File)file, (CharSequence)"this is brand new data", (boolean)true);
        String expected = "This line was there before you...this is brand new data";
        String actual = FileUtils.readFileToString((File)file);
        Assert.assertEquals((Object)"This line was there before you...this is brand new data", (Object)actual);
    }

    @Test
    public void testWrite_WithAppendOptionFalse_ShouldDeletePreviousFileLines() throws Exception {
        File file = TestUtils.newFile(this.getTestDirectory(), "lines.txt");
        FileUtils.writeStringToFile((File)file, (String)"This line was there before you...");
        FileUtils.write((File)file, (CharSequence)"this is brand new data", (boolean)false);
        String expected = "this is brand new data";
        String actual = FileUtils.readFileToString((File)file);
        Assert.assertEquals((Object)"this is brand new data", (Object)actual);
    }

    @Test
    public void testWriteByteArrayToFile_WithAppendOptionTrue_ShouldNotDeletePreviousFileLines() throws Exception {
        File file = TestUtils.newFile(this.getTestDirectory(), "lines.txt");
        FileUtils.writeStringToFile((File)file, (String)"This line was there before you...");
        FileUtils.writeByteArrayToFile((File)file, (byte[])"this is brand new data".getBytes(), (boolean)true);
        String expected = "This line was there before you...this is brand new data";
        String actual = FileUtils.readFileToString((File)file);
        Assert.assertEquals((Object)"This line was there before you...this is brand new data", (Object)actual);
    }

    @Test
    public void testWriteByteArrayToFile_WithAppendOptionFalse_ShouldDeletePreviousFileLines() throws Exception {
        File file = TestUtils.newFile(this.getTestDirectory(), "lines.txt");
        FileUtils.writeStringToFile((File)file, (String)"This line was there before you...");
        FileUtils.writeByteArrayToFile((File)file, (byte[])"this is brand new data".getBytes(), (boolean)false);
        String expected = "this is brand new data";
        String actual = FileUtils.readFileToString((File)file);
        Assert.assertEquals((Object)"this is brand new data", (Object)actual);
    }

    @Test
    public void testWriteByteArrayToFile_WithOffsetAndLength_WithAppendOptionTrue_ShouldNotDeletePreviousFileLines() throws Exception {
        File file = TestUtils.newFile(this.getTestDirectory(), "lines.txt");
        FileUtils.writeStringToFile((File)file, (String)"This line was there before you...");
        byte[] data = "SKIP_THIS_this is brand new data_AND_SKIP_THIS".getBytes(Charsets.UTF_8);
        FileUtils.writeByteArrayToFile((File)file, (byte[])data, (int)10, (int)22, (boolean)true);
        String expected = "This line was there before you...this is brand new data";
        String actual = FileUtils.readFileToString((File)file, (Charset)Charsets.UTF_8);
        Assert.assertEquals((Object)"This line was there before you...this is brand new data", (Object)actual);
    }

    @Test
    public void testWriteByteArrayToFile_WithOffsetAndLength_WithAppendOptionTrue_ShouldDeletePreviousFileLines() throws Exception {
        File file = TestUtils.newFile(this.getTestDirectory(), "lines.txt");
        FileUtils.writeStringToFile((File)file, (String)"This line was there before you...");
        byte[] data = "SKIP_THIS_this is brand new data_AND_SKIP_THIS".getBytes(Charsets.UTF_8);
        FileUtils.writeByteArrayToFile((File)file, (byte[])data, (int)10, (int)22, (boolean)false);
        String expected = "this is brand new data";
        String actual = FileUtils.readFileToString((File)file, (Charset)Charsets.UTF_8);
        Assert.assertEquals((Object)"this is brand new data", (Object)actual);
    }

    @Test
    public void testChecksumCRC32() throws Exception {
        String text = "Imagination is more important than knowledge - Einstein";
        File file = new File(this.getTestDirectory(), "checksum-test.txt");
        FileUtils.writeStringToFile((File)file, (String)"Imagination is more important than knowledge - Einstein", (String)"US-ASCII");
        CRC32 expectedChecksum = new CRC32();
        expectedChecksum.update("Imagination is more important than knowledge - Einstein".getBytes("US-ASCII"), 0, "Imagination is more important than knowledge - Einstein".length());
        long expectedValue = expectedChecksum.getValue();
        long resultValue = FileUtils.checksumCRC32((File)file);
        Assert.assertEquals((long)expectedValue, (long)resultValue);
    }

    @Test
    public void testChecksum() throws Exception {
        String text = "Imagination is more important than knowledge - Einstein";
        File file = new File(this.getTestDirectory(), "checksum-test.txt");
        FileUtils.writeStringToFile((File)file, (String)"Imagination is more important than knowledge - Einstein", (String)"US-ASCII");
        CRC32 expectedChecksum = new CRC32();
        expectedChecksum.update("Imagination is more important than knowledge - Einstein".getBytes("US-ASCII"), 0, "Imagination is more important than knowledge - Einstein".length());
        long expectedValue = expectedChecksum.getValue();
        CRC32 testChecksum = new CRC32();
        Checksum resultChecksum = FileUtils.checksum((File)file, (Checksum)testChecksum);
        long resultValue = resultChecksum.getValue();
        Assert.assertSame((Object)testChecksum, (Object)resultChecksum);
        Assert.assertEquals((long)expectedValue, (long)resultValue);
    }

    @Test
    public void testChecksumOnNullFile() throws Exception {
        try {
            FileUtils.checksum(null, (Checksum)new CRC32());
            Assert.fail();
        }
        catch (NullPointerException nullPointerException) {
            // empty catch block
        }
    }

    @Test
    public void testChecksumOnNullChecksum() throws Exception {
        String text = "Imagination is more important than knowledge - Einstein";
        File file = new File(this.getTestDirectory(), "checksum-test.txt");
        FileUtils.writeStringToFile((File)file, (String)"Imagination is more important than knowledge - Einstein", (String)"US-ASCII");
        try {
            FileUtils.checksum((File)file, null);
            Assert.fail();
        }
        catch (NullPointerException nullPointerException) {
            // empty catch block
        }
    }

    @Test
    public void testChecksumOnDirectory() throws Exception {
        try {
            FileUtils.checksum((File)new File("."), (Checksum)new CRC32());
            Assert.fail();
        }
        catch (IllegalArgumentException illegalArgumentException) {
            // empty catch block
        }
    }

    @Test
    public void testChecksumDouble() throws Exception {
        String text1 = "Imagination is more important than knowledge - Einstein";
        File file1 = new File(this.getTestDirectory(), "checksum-test.txt");
        FileUtils.writeStringToFile((File)file1, (String)"Imagination is more important than knowledge - Einstein", (String)"US-ASCII");
        String text2 = "To be or not to be - Shakespeare";
        File file2 = new File(this.getTestDirectory(), "checksum-test2.txt");
        FileUtils.writeStringToFile((File)file2, (String)"To be or not to be - Shakespeare", (String)"US-ASCII");
        CRC32 expectedChecksum = new CRC32();
        expectedChecksum.update("Imagination is more important than knowledge - Einstein".getBytes("US-ASCII"), 0, "Imagination is more important than knowledge - Einstein".length());
        expectedChecksum.update("To be or not to be - Shakespeare".getBytes("US-ASCII"), 0, "To be or not to be - Shakespeare".length());
        long expectedValue = expectedChecksum.getValue();
        CRC32 testChecksum = new CRC32();
        FileUtils.checksum((File)file1, (Checksum)testChecksum);
        FileUtils.checksum((File)file2, (Checksum)testChecksum);
        long resultValue = testChecksum.getValue();
        Assert.assertEquals((long)expectedValue, (long)resultValue);
    }

    @Test
    public void testDeleteDirectoryWithNonDirectory() throws Exception {
        try {
            FileUtils.deleteDirectory((File)this.testFile1);
            Assert.fail();
        }
        catch (IllegalArgumentException illegalArgumentException) {
            // empty catch block
        }
    }

    @Test
    public void testDeleteQuietlyForNull() {
        try {
            FileUtils.deleteQuietly(null);
        }
        catch (Exception ex) {
            Assert.fail((String)ex.getMessage());
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    public void testDeleteQuietlyDir() throws IOException {
        File testDirectory = new File(this.getTestDirectory(), "testDeleteQuietlyDir");
        File testFile = new File(testDirectory, "testDeleteQuietlyFile");
        testDirectory.mkdirs();
        if (!testFile.getParentFile().exists()) {
            throw new IOException("Cannot create file " + testFile + " as the parent directory does not exist");
        }
        BufferedOutputStream output = new BufferedOutputStream(new FileOutputStream(testFile));
        try {
            TestUtils.generateTestData(output, 0L);
        }
        finally {
            IOUtils.closeQuietly((OutputStream)output);
        }
        Assert.assertTrue((boolean)testDirectory.exists());
        Assert.assertTrue((boolean)testFile.exists());
        FileUtils.deleteQuietly((File)testDirectory);
        Assert.assertFalse((String)"Check No Exist", (boolean)testDirectory.exists());
        Assert.assertFalse((String)"Check No Exist", (boolean)testFile.exists());
    }

    @Test
    public void testDeleteQuietlyFile() throws IOException {
        File testFile = new File(this.getTestDirectory(), "testDeleteQuietlyFile");
        if (!testFile.getParentFile().exists()) {
            throw new IOException("Cannot create file " + testFile + " as the parent directory does not exist");
        }
        BufferedOutputStream output = new BufferedOutputStream(new FileOutputStream(testFile));
        try {
            TestUtils.generateTestData(output, 0L);
        }
        finally {
            IOUtils.closeQuietly((OutputStream)output);
        }
        Assert.assertTrue((boolean)testFile.exists());
        FileUtils.deleteQuietly((File)testFile);
        Assert.assertFalse((String)"Check No Exist", (boolean)testFile.exists());
    }

    @Test
    public void testDeleteQuietlyNonExistent() {
        File testFile = new File("testDeleteQuietlyNonExistent");
        Assert.assertFalse((boolean)testFile.exists());
        try {
            FileUtils.deleteQuietly((File)testFile);
        }
        catch (Exception ex) {
            Assert.fail((String)ex.getMessage());
        }
    }

    @Test
    public void testMoveFile_Rename() throws Exception {
        File destination = new File(this.getTestDirectory(), "move1.txt");
        FileUtils.moveFile((File)this.testFile1, (File)destination);
        Assert.assertTrue((String)"Check Exist", (boolean)destination.exists());
        Assert.assertTrue((String)"Original deleted", (!this.testFile1.exists() ? 1 : 0) != 0);
    }

    @Test
    public void testMoveFile_CopyDelete() throws Exception {
        File destination = new File(this.getTestDirectory(), "move2.txt");
        File src = new File(this.testFile1.getAbsolutePath()){
            private static final long serialVersionUID = 1L;

            @Override
            public boolean renameTo(File f) {
                return false;
            }
        };
        FileUtils.moveFile((File)src, (File)destination);
        Assert.assertTrue((String)"Check Exist", (boolean)destination.exists());
        Assert.assertTrue((String)"Original deleted", (!src.exists() ? 1 : 0) != 0);
    }

    @Test
    public void testMoveFile_CopyDelete_Failed() throws Exception {
        File destination = new File(this.getTestDirectory(), "move3.txt");
        File src = new File(this.testFile1.getAbsolutePath()){
            private static final long serialVersionUID = 1L;

            @Override
            public boolean renameTo(File f) {
                return false;
            }

            @Override
            public boolean delete() {
                return false;
            }
        };
        try {
            FileUtils.moveFile((File)src, (File)destination);
            Assert.fail((String)"move should have failed as src has not been deleted");
        }
        catch (IOException e) {
            Assert.assertTrue((String)"Check Rollback", (!destination.exists() ? 1 : 0) != 0);
            Assert.assertTrue((String)"Original exists", (boolean)src.exists());
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    public void testMoveFile_Errors() throws Exception {
        try {
            FileUtils.moveFile(null, (File)new File("foo"));
            Assert.fail((String)"Expected NullPointerException when source is null");
        }
        catch (NullPointerException nullPointerException) {
            // empty catch block
        }
        try {
            FileUtils.moveFile((File)new File("foo"), null);
            Assert.fail((String)"Expected NullPointerException when destination is null");
        }
        catch (NullPointerException nullPointerException) {
            // empty catch block
        }
        try {
            FileUtils.moveFile((File)new File("nonexistant"), (File)new File("foo"));
            Assert.fail((String)"Expected FileNotFoundException for source");
        }
        catch (FileNotFoundException fileNotFoundException) {
            // empty catch block
        }
        try {
            FileUtils.moveFile((File)this.getTestDirectory(), (File)new File("foo"));
            Assert.fail((String)"Expected IOException when source is a directory");
        }
        catch (IOException iOException) {
            // empty catch block
        }
        File testSourceFile = new File(this.getTestDirectory(), "testMoveFileSource");
        File testDestFile = new File(this.getTestDirectory(), "testMoveFileSource");
        if (!testSourceFile.getParentFile().exists()) {
            throw new IOException("Cannot create file " + testSourceFile + " as the parent directory does not exist");
        }
        BufferedOutputStream output1 = new BufferedOutputStream(new FileOutputStream(testSourceFile));
        try {
            TestUtils.generateTestData(output1, 0L);
        }
        finally {
            IOUtils.closeQuietly((OutputStream)output1);
        }
        if (!testDestFile.getParentFile().exists()) {
            throw new IOException("Cannot create file " + testDestFile + " as the parent directory does not exist");
        }
        BufferedOutputStream output = new BufferedOutputStream(new FileOutputStream(testDestFile));
        try {
            TestUtils.generateTestData(output, 0L);
        }
        finally {
            IOUtils.closeQuietly((OutputStream)output);
        }
        try {
            FileUtils.moveFile((File)testSourceFile, (File)testDestFile);
            Assert.fail((String)"Expected FileExistsException when dest already exists");
        }
        catch (FileExistsException fileExistsException) {
            // empty catch block
        }
    }

    @Test
    public void testMoveFileToDirectory() throws Exception {
        File destDir = new File(this.getTestDirectory(), "moveFileDestDir");
        File movedFile = new File(destDir, this.testFile1.getName());
        Assert.assertFalse((String)"Check Exist before", (boolean)destDir.exists());
        Assert.assertFalse((String)"Check Exist before", (boolean)movedFile.exists());
        FileUtils.moveFileToDirectory((File)this.testFile1, (File)destDir, (boolean)true);
        Assert.assertTrue((String)"Check Exist after", (boolean)movedFile.exists());
        Assert.assertTrue((String)"Original deleted", (!this.testFile1.exists() ? 1 : 0) != 0);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    public void testMoveFileToDirectory_Errors() throws Exception {
        try {
            FileUtils.moveFileToDirectory(null, (File)new File("foo"), (boolean)true);
            Assert.fail((String)"Expected NullPointerException when source is null");
        }
        catch (NullPointerException nullPointerException) {
            // empty catch block
        }
        try {
            FileUtils.moveFileToDirectory((File)new File("foo"), null, (boolean)true);
            Assert.fail((String)"Expected NullPointerException when destination is null");
        }
        catch (NullPointerException nullPointerException) {
            // empty catch block
        }
        File testFile1 = new File(this.getTestDirectory(), "testMoveFileFile1");
        File testFile2 = new File(this.getTestDirectory(), "testMoveFileFile2");
        if (!testFile1.getParentFile().exists()) {
            throw new IOException("Cannot create file " + testFile1 + " as the parent directory does not exist");
        }
        BufferedOutputStream output1 = new BufferedOutputStream(new FileOutputStream(testFile1));
        try {
            TestUtils.generateTestData(output1, 0L);
        }
        finally {
            IOUtils.closeQuietly((OutputStream)output1);
        }
        if (!testFile2.getParentFile().exists()) {
            throw new IOException("Cannot create file " + testFile2 + " as the parent directory does not exist");
        }
        BufferedOutputStream output = new BufferedOutputStream(new FileOutputStream(testFile2));
        try {
            TestUtils.generateTestData(output, 0L);
        }
        finally {
            IOUtils.closeQuietly((OutputStream)output);
        }
        try {
            FileUtils.moveFileToDirectory((File)testFile1, (File)testFile2, (boolean)true);
            Assert.fail((String)"Expected IOException when dest not a directory");
        }
        catch (IOException iOException) {
            // empty catch block
        }
        File nonexistant = new File(this.getTestDirectory(), "testMoveFileNonExistant");
        try {
            FileUtils.moveFileToDirectory((File)testFile1, (File)nonexistant, (boolean)false);
            Assert.fail((String)"Expected IOException when dest does not exist and create=false");
        }
        catch (IOException iOException) {
            // empty catch block
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    public void testMoveDirectory_Rename() throws Exception {
        File dir = this.getTestDirectory();
        File src = new File(dir, "testMoveDirectory1Source");
        File testDir = new File(src, "foo");
        File testFile = new File(testDir, "bar");
        testDir.mkdirs();
        if (!testFile.getParentFile().exists()) {
            throw new IOException("Cannot create file " + testFile + " as the parent directory does not exist");
        }
        BufferedOutputStream output = new BufferedOutputStream(new FileOutputStream(testFile));
        try {
            TestUtils.generateTestData(output, 0L);
        }
        finally {
            IOUtils.closeQuietly((OutputStream)output);
        }
        File destination = new File(dir, "testMoveDirectory1Dest");
        FileUtils.deleteDirectory((File)destination);
        FileUtils.moveDirectory((File)src, (File)destination);
        Assert.assertTrue((String)"Check Exist", (boolean)destination.exists());
        Assert.assertTrue((String)"Original deleted", (!src.exists() ? 1 : 0) != 0);
        File movedDir = new File(destination, testDir.getName());
        File movedFile = new File(movedDir, testFile.getName());
        Assert.assertTrue((String)"Check dir moved", (boolean)movedDir.exists());
        Assert.assertTrue((String)"Check file moved", (boolean)movedFile.exists());
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    public void testMoveDirectory_CopyDelete() throws Exception {
        File dir = this.getTestDirectory();
        File src = new File(dir, "testMoveDirectory2Source"){
            private static final long serialVersionUID = 1L;

            @Override
            public boolean renameTo(File dest) {
                return false;
            }
        };
        File testDir = new File(src, "foo");
        File testFile = new File(testDir, "bar");
        testDir.mkdirs();
        if (!testFile.getParentFile().exists()) {
            throw new IOException("Cannot create file " + testFile + " as the parent directory does not exist");
        }
        BufferedOutputStream output = new BufferedOutputStream(new FileOutputStream(testFile));
        try {
            TestUtils.generateTestData(output, 0L);
        }
        finally {
            IOUtils.closeQuietly((OutputStream)output);
        }
        File destination = new File(dir, "testMoveDirectory1Dest");
        FileUtils.deleteDirectory((File)destination);
        FileUtils.moveDirectory((File)src, (File)destination);
        Assert.assertTrue((String)"Check Exist", (boolean)destination.exists());
        Assert.assertTrue((String)"Original deleted", (!src.exists() ? 1 : 0) != 0);
        File movedDir = new File(destination, testDir.getName());
        File movedFile = new File(movedDir, testFile.getName());
        Assert.assertTrue((String)"Check dir moved", (boolean)movedDir.exists());
        Assert.assertTrue((String)"Check file moved", (boolean)movedFile.exists());
    }

    @Test
    public void testMoveDirectory_Errors() throws Exception {
        try {
            FileUtils.moveDirectory(null, (File)new File("foo"));
            Assert.fail((String)"Expected NullPointerException when source is null");
        }
        catch (NullPointerException nullPointerException) {
            // empty catch block
        }
        try {
            FileUtils.moveDirectory((File)new File("foo"), null);
            Assert.fail((String)"Expected NullPointerException when destination is null");
        }
        catch (NullPointerException nullPointerException) {
            // empty catch block
        }
        try {
            FileUtils.moveDirectory((File)new File("nonexistant"), (File)new File("foo"));
            Assert.fail((String)"Expected FileNotFoundException for source");
        }
        catch (FileNotFoundException fileNotFoundException) {
            // empty catch block
        }
        File testFile = new File(this.getTestDirectory(), "testMoveDirectoryFile");
        if (!testFile.getParentFile().exists()) {
            throw new IOException("Cannot create file " + testFile + " as the parent directory does not exist");
        }
        BufferedOutputStream output = new BufferedOutputStream(new FileOutputStream(testFile));
        try {
            TestUtils.generateTestData(output, 0L);
        }
        finally {
            IOUtils.closeQuietly((OutputStream)output);
        }
        try {
            FileUtils.moveDirectory((File)testFile, (File)new File("foo"));
            Assert.fail((String)"Expected IOException when source is not a directory");
        }
        catch (IOException iOException) {
            // empty catch block
        }
        File testSrcFile = new File(this.getTestDirectory(), "testMoveDirectorySource");
        File testDestFile = new File(this.getTestDirectory(), "testMoveDirectoryDest");
        testSrcFile.mkdir();
        testDestFile.mkdir();
        try {
            FileUtils.moveDirectory((File)testSrcFile, (File)testDestFile);
            Assert.fail((String)"Expected FileExistsException when dest already exists");
        }
        catch (FileExistsException fileExistsException) {
            // empty catch block
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    public void testMoveDirectoryToDirectory() throws Exception {
        File dir = this.getTestDirectory();
        File src = new File(dir, "testMoveDirectory1Source");
        File testChildDir = new File(src, "foo");
        File testFile = new File(testChildDir, "bar");
        testChildDir.mkdirs();
        if (!testFile.getParentFile().exists()) {
            throw new IOException("Cannot create file " + testFile + " as the parent directory does not exist");
        }
        BufferedOutputStream output = new BufferedOutputStream(new FileOutputStream(testFile));
        try {
            TestUtils.generateTestData(output, 0L);
        }
        finally {
            IOUtils.closeQuietly((OutputStream)output);
        }
        File destDir = new File(dir, "testMoveDirectory1Dest");
        FileUtils.deleteDirectory((File)destDir);
        Assert.assertFalse((String)"Check Exist before", (boolean)destDir.exists());
        FileUtils.moveDirectoryToDirectory((File)src, (File)destDir, (boolean)true);
        Assert.assertTrue((String)"Check Exist after", (boolean)destDir.exists());
        Assert.assertTrue((String)"Original deleted", (!src.exists() ? 1 : 0) != 0);
        File movedDir = new File(destDir, src.getName());
        File movedChildDir = new File(movedDir, testChildDir.getName());
        File movedFile = new File(movedChildDir, testFile.getName());
        Assert.assertTrue((String)"Check dir moved", (boolean)movedDir.exists());
        Assert.assertTrue((String)"Check child dir moved", (boolean)movedChildDir.exists());
        Assert.assertTrue((String)"Check file moved", (boolean)movedFile.exists());
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    public void testMoveDirectoryToDirectory_Errors() throws Exception {
        try {
            FileUtils.moveDirectoryToDirectory(null, (File)new File("foo"), (boolean)true);
            Assert.fail((String)"Expected NullPointerException when source is null");
        }
        catch (NullPointerException nullPointerException) {
            // empty catch block
        }
        try {
            FileUtils.moveDirectoryToDirectory((File)new File("foo"), null, (boolean)true);
            Assert.fail((String)"Expected NullPointerException when destination is null");
        }
        catch (NullPointerException nullPointerException) {
            // empty catch block
        }
        File testFile1 = new File(this.getTestDirectory(), "testMoveFileFile1");
        File testFile2 = new File(this.getTestDirectory(), "testMoveFileFile2");
        if (!testFile1.getParentFile().exists()) {
            throw new IOException("Cannot create file " + testFile1 + " as the parent directory does not exist");
        }
        BufferedOutputStream output1 = new BufferedOutputStream(new FileOutputStream(testFile1));
        try {
            TestUtils.generateTestData(output1, 0L);
        }
        finally {
            IOUtils.closeQuietly((OutputStream)output1);
        }
        if (!testFile2.getParentFile().exists()) {
            throw new IOException("Cannot create file " + testFile2 + " as the parent directory does not exist");
        }
        BufferedOutputStream output = new BufferedOutputStream(new FileOutputStream(testFile2));
        try {
            TestUtils.generateTestData(output, 0L);
        }
        finally {
            IOUtils.closeQuietly((OutputStream)output);
        }
        try {
            FileUtils.moveDirectoryToDirectory((File)testFile1, (File)testFile2, (boolean)true);
            Assert.fail((String)"Expected IOException when dest not a directory");
        }
        catch (IOException iOException) {
            // empty catch block
        }
        File nonexistant = new File(this.getTestDirectory(), "testMoveFileNonExistant");
        try {
            FileUtils.moveDirectoryToDirectory((File)testFile1, (File)nonexistant, (boolean)false);
            Assert.fail((String)"Expected IOException when dest does not exist and create=false");
        }
        catch (IOException iOException) {
            // empty catch block
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    public void testMoveToDirectory() throws Exception {
        File destDir = new File(this.getTestDirectory(), "testMoveToDirectoryDestDir");
        File testDir = new File(this.getTestDirectory(), "testMoveToDirectoryTestDir");
        File testFile = new File(this.getTestDirectory(), "testMoveToDirectoryTestFile");
        testDir.mkdirs();
        if (!testFile.getParentFile().exists()) {
            throw new IOException("Cannot create file " + testFile + " as the parent directory does not exist");
        }
        BufferedOutputStream output = new BufferedOutputStream(new FileOutputStream(testFile));
        try {
            TestUtils.generateTestData(output, 0L);
        }
        finally {
            IOUtils.closeQuietly((OutputStream)output);
        }
        File movedFile = new File(destDir, testFile.getName());
        File movedDir = new File(destDir, testFile.getName());
        Assert.assertFalse((String)"Check File Doesnt exist", (boolean)movedFile.exists());
        Assert.assertFalse((String)"Check Dir Doesnt exist", (boolean)movedDir.exists());
        FileUtils.moveToDirectory((File)testFile, (File)destDir, (boolean)true);
        Assert.assertTrue((String)"Check File exists", (boolean)movedFile.exists());
        Assert.assertFalse((String)"Check Original File doesn't exist", (boolean)testFile.exists());
        FileUtils.moveToDirectory((File)testDir, (File)destDir, (boolean)true);
        Assert.assertTrue((String)"Check Dir exists", (boolean)movedDir.exists());
        Assert.assertFalse((String)"Check Original Dir doesn't exist", (boolean)testDir.exists());
    }

    @Test
    public void testMoveToDirectory_Errors() throws Exception {
        try {
            FileUtils.moveDirectoryToDirectory(null, (File)new File("foo"), (boolean)true);
            Assert.fail((String)"Expected NullPointerException when source is null");
        }
        catch (NullPointerException nullPointerException) {
            // empty catch block
        }
        try {
            FileUtils.moveDirectoryToDirectory((File)new File("foo"), null, (boolean)true);
            Assert.fail((String)"Expected NullPointerException when destination is null");
        }
        catch (NullPointerException nullPointerException) {
            // empty catch block
        }
        File nonexistant = new File(this.getTestDirectory(), "nonexistant");
        File destDir = new File(this.getTestDirectory(), "MoveToDirectoryDestDir");
        try {
            FileUtils.moveToDirectory((File)nonexistant, (File)destDir, (boolean)true);
            Assert.fail((String)"Expected IOException when source does not exist");
        }
        catch (IOException iOException) {
            // empty catch block
        }
    }

    @Test
    public void testIO300() throws Exception {
        File testDirectory = this.getTestDirectory();
        File src = new File(testDirectory, "dir1");
        File dest = new File(src, "dir2");
        Assert.assertTrue((boolean)dest.mkdirs());
        Assert.assertTrue((boolean)src.exists());
        try {
            FileUtils.moveDirectoryToDirectory((File)src, (File)dest, (boolean)false);
            Assert.fail((String)"expected IOException");
        }
        catch (IOException iOException) {
            // empty catch block
        }
        Assert.assertTrue((boolean)src.exists());
    }

    @Test
    public void testIO276() throws Exception {
        File dir = new File("target", "IO276");
        Assert.assertTrue((String)(dir + " should not be present"), (boolean)dir.mkdirs());
        File file = new File(dir, "IO276.txt");
        Assert.assertTrue((String)(file + " should not be present"), (boolean)file.createNewFile());
        FileUtils.forceDeleteOnExit((File)dir);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    public void testIncorrectOutputSize() throws Exception {
        File inFile = new File("pom.xml");
        ShorterFile outFile = new ShorterFile("target/pom.tmp");
        try {
            FileUtils.copyFile((File)inFile, (File)outFile);
            Assert.fail((String)"Expected IOException");
        }
        catch (Exception e) {
            String msg = e.toString();
            Assert.assertTrue((String)msg, (boolean)msg.contains("Failed to copy full contents"));
        }
        finally {
            outFile.delete();
        }
    }

    static class ListDirectoryWalker
    extends DirectoryWalker<File> {
        ListDirectoryWalker() {
        }

        List<File> list(File startDirectory) throws IOException {
            ArrayList<File> files = new ArrayList<File>();
            this.walk(startDirectory, files);
            return files;
        }

        protected void handleDirectoryStart(File directory, int depth, Collection<File> results) throws IOException {
            if (depth > 0) {
                results.add(directory);
            }
        }

        protected void handleFile(File file, int depth, Collection<File> results) throws IOException {
            results.add(file);
        }
    }

    private static class ShorterFile
    extends File {
        private static final long serialVersionUID = 1L;

        public ShorterFile(String pathname) {
            super(pathname);
        }

        @Override
        public long length() {
            return super.length() - 1L;
        }
    }
}

