/*
 * Decompiled with CFR 0.152.
 */
package org.apache.commons.compress.utils;

import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.channels.ClosedChannelException;
import java.nio.channels.NonWritableChannelException;
import java.nio.channels.SeekableByteChannel;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Arrays;
import org.apache.commons.compress.utils.MultiReadOnlySeekableByteChannel;
import org.apache.commons.compress.utils.SeekableInMemoryByteChannel;
import org.junit.Assert;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.ExpectedException;

public class MultiReadOnlySeekableByteChannelTest {
    @Rule
    public ExpectedException thrown = ExpectedException.none();

    @Test
    public void constructorThrowsOnNullArg() {
        this.thrown.expect(NullPointerException.class);
        new MultiReadOnlySeekableByteChannel(null);
    }

    @Test
    public void forSeekableByteChannelsThrowsOnNullArg() {
        this.thrown.expect(NullPointerException.class);
        MultiReadOnlySeekableByteChannel.forSeekableByteChannels(null);
    }

    @Test
    public void forFilesThrowsOnNullArg() throws IOException {
        this.thrown.expect(NullPointerException.class);
        MultiReadOnlySeekableByteChannel.forFiles(null);
    }

    @Test
    public void forSeekableByteChannelsReturnsIdentityForSingleElement() {
        SeekableByteChannel e = this.makeEmpty();
        SeekableByteChannel m = MultiReadOnlySeekableByteChannel.forSeekableByteChannels((SeekableByteChannel[])new SeekableByteChannel[]{e});
        Assert.assertSame((Object)e, (Object)m);
    }

    @Test
    public void referenceBehaviorForEmptyChannel() throws IOException {
        this.checkEmpty(this.makeEmpty());
    }

    @Test
    public void twoEmptyChannelsConcatenateAsEmptyChannel() throws IOException {
        this.checkEmpty(MultiReadOnlySeekableByteChannel.forSeekableByteChannels((SeekableByteChannel[])new SeekableByteChannel[]{this.makeEmpty(), this.makeEmpty()}));
    }

    @Test
    public void checkForSingleByte() throws IOException {
        this.check(new byte[]{0});
    }

    @Test
    public void checkForString() throws IOException {
        this.check("abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789".getBytes(StandardCharsets.UTF_8));
    }

    @Test
    public void verifyGrouped() {
        Assert.assertArrayEquals((Object[])new byte[][]{{1, 2, 3}, {4, 5, 6}, {7}}, (Object[])this.grouped(new byte[]{1, 2, 3, 4, 5, 6, 7}, 3));
        Assert.assertArrayEquals((Object[])new byte[][]{{1, 2, 3}, {4, 5, 6}}, (Object[])this.grouped(new byte[]{1, 2, 3, 4, 5, 6}, 3));
        Assert.assertArrayEquals((Object[])new byte[][]{{1, 2, 3}, {4, 5}}, (Object[])this.grouped(new byte[]{1, 2, 3, 4, 5}, 3));
    }

    @Test
    public void closesAllAndThrowsExceptionIfCloseThrows() {
        SeekableByteChannel[] ts = new ThrowingSeekableByteChannel[]{new ThrowingSeekableByteChannel(), new ThrowingSeekableByteChannel()};
        SeekableByteChannel s = MultiReadOnlySeekableByteChannel.forSeekableByteChannels((SeekableByteChannel[])ts);
        try {
            s.close();
            Assert.fail((String)"IOException expected");
        }
        catch (IOException iOException) {
            // empty catch block
        }
        Assert.assertFalse((boolean)ts[0].isOpen());
        Assert.assertFalse((boolean)ts[1].isOpen());
    }

    @Test
    public void cantTruncate() throws IOException {
        SeekableByteChannel s = MultiReadOnlySeekableByteChannel.forSeekableByteChannels((SeekableByteChannel[])new SeekableByteChannel[]{this.makeEmpty(), this.makeEmpty()});
        this.thrown.expect(NonWritableChannelException.class);
        s.truncate(1L);
    }

    @Test
    public void cantWrite() throws IOException {
        SeekableByteChannel s = MultiReadOnlySeekableByteChannel.forSeekableByteChannels((SeekableByteChannel[])new SeekableByteChannel[]{this.makeEmpty(), this.makeEmpty()});
        this.thrown.expect(NonWritableChannelException.class);
        s.write(ByteBuffer.allocate(10));
    }

    @Test
    public void cantPositionToANegativePosition() throws IOException {
        SeekableByteChannel s = MultiReadOnlySeekableByteChannel.forSeekableByteChannels((SeekableByteChannel[])new SeekableByteChannel[]{this.makeEmpty(), this.makeEmpty()});
        this.thrown.expect(IllegalArgumentException.class);
        s.position(-1L);
    }

    private SeekableByteChannel makeEmpty() {
        return this.makeSingle(new byte[0]);
    }

    private SeekableByteChannel makeSingle(byte[] arr) {
        return new SeekableInMemoryByteChannel(arr);
    }

    private SeekableByteChannel makeMulti(byte[][] arr) {
        SeekableByteChannel[] s = new SeekableByteChannel[arr.length];
        for (int i = 0; i < s.length; ++i) {
            s[i] = this.makeSingle(arr[i]);
        }
        return MultiReadOnlySeekableByteChannel.forSeekableByteChannels((SeekableByteChannel[])s);
    }

    private void checkEmpty(SeekableByteChannel channel) throws IOException {
        ByteBuffer buf = ByteBuffer.allocate(10);
        Assert.assertTrue((boolean)channel.isOpen());
        Assert.assertEquals((long)0L, (long)channel.size());
        Assert.assertEquals((long)0L, (long)channel.position());
        Assert.assertEquals((long)-1L, (long)channel.read(buf));
        channel.position(5L);
        Assert.assertEquals((long)-1L, (long)channel.read(buf));
        channel.close();
        Assert.assertFalse((boolean)channel.isOpen());
        try {
            channel.read(buf);
            Assert.fail((String)"expected a ClosedChannelException");
        }
        catch (ClosedChannelException closedChannelException) {
            // empty catch block
        }
        try {
            channel.position(100L);
            Assert.fail((String)"expected a ClosedChannelException");
        }
        catch (ClosedChannelException closedChannelException) {
            // empty catch block
        }
    }

    private void check(byte[] expected) throws IOException {
        for (int channelSize = 1; channelSize <= expected.length; ++channelSize) {
            this.check(expected, this.makeSingle(expected));
            this.check(expected, this.makeMulti(this.grouped(expected, channelSize)));
        }
    }

    private void check(byte[] expected, SeekableByteChannel channel) throws IOException {
        for (int readBufferSize = 1; readBufferSize <= expected.length + 5; ++readBufferSize) {
            this.check(expected, channel, readBufferSize);
        }
    }

    private void check(byte[] expected, SeekableByteChannel channel, int readBufferSize) throws IOException {
        Assert.assertTrue((String)("readBufferSize " + readBufferSize), (boolean)channel.isOpen());
        Assert.assertEquals((String)("readBufferSize " + readBufferSize), (long)expected.length, (long)channel.size());
        channel.position(0L);
        Assert.assertEquals((String)("readBufferSize " + readBufferSize), (long)0L, (long)channel.position());
        Assert.assertEquals((String)("readBufferSize " + readBufferSize), (long)0L, (long)channel.read(ByteBuffer.allocate(0)));
        ByteBuffer resultBuffer = ByteBuffer.allocate(expected.length + 100);
        ByteBuffer buf = ByteBuffer.allocate(readBufferSize);
        int bytesRead = channel.read(buf);
        while (bytesRead != -1) {
            int remaining = buf.remaining();
            buf.flip();
            resultBuffer.put(buf);
            buf.clear();
            bytesRead = channel.read(buf);
            if (resultBuffer.position() < expected.length) {
                Assert.assertEquals((String)("readBufferSize " + readBufferSize), (long)0L, (long)remaining);
            }
            if (bytesRead == -1) {
                Assert.assertEquals((String)("readBufferSize " + readBufferSize), (long)0L, (long)buf.position());
                continue;
            }
            Assert.assertEquals((String)("readBufferSize " + readBufferSize), (long)bytesRead, (long)buf.position());
        }
        resultBuffer.flip();
        byte[] arr = new byte[resultBuffer.remaining()];
        resultBuffer.get(arr);
        Assert.assertArrayEquals((String)("readBufferSize " + readBufferSize), (byte[])expected, (byte[])arr);
    }

    private byte[][] grouped(byte[] input, int chunkSize) {
        ArrayList<byte[]> groups = new ArrayList<byte[]>();
        int idx = 0;
        while (idx + chunkSize <= input.length) {
            groups.add(Arrays.copyOfRange(input, idx, idx + chunkSize));
            idx += chunkSize;
        }
        if (idx < input.length) {
            groups.add(Arrays.copyOfRange(input, idx, input.length));
        }
        return (byte[][])groups.toArray((T[])new byte[0][]);
    }

    private static class ThrowingSeekableByteChannel
    implements SeekableByteChannel {
        private boolean closed = false;

        private ThrowingSeekableByteChannel() {
        }

        @Override
        public boolean isOpen() {
            return !this.closed;
        }

        @Override
        public void close() throws IOException {
            this.closed = true;
            throw new IOException("foo");
        }

        @Override
        public int read(ByteBuffer dst) throws IOException {
            return -1;
        }

        @Override
        public int write(ByteBuffer src) throws IOException {
            return 0;
        }

        @Override
        public long position() throws IOException {
            return 0L;
        }

        @Override
        public SeekableByteChannel position(long newPosition) throws IOException {
            return this;
        }

        @Override
        public long size() throws IOException {
            return 0L;
        }

        @Override
        public SeekableByteChannel truncate(long size) throws IOException {
            return this;
        }
    }
}

